# -*- coding: utf-8 -*-
import os

import sandbox.common.types.client as ctc

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.sandboxsdk.parameters import SandboxStringParameter


class ArcadiaUrlParameter(SandboxStringParameter):
    name = 'arcadia_branch'
    description = 'Svn url for arcadia'
    required = True
    default_value = 'arcadia:/arc/trunk/arcadia/'


def branch_name_from_path(path):
    components = path.split('/')

    def look_for(what):
        if what in components:
            return components[components.index(what) + 2]
        return None
    return look_for('branches') or look_for('tags') or 'trunk'


class AnnotateCGI(SandboxTask):
    """
    Таск построения документации на wiki CGI-параметров и других ручек.
    Таск формирует страничку `с описанием CGI-параметров базового поиска <https://wiki.yandex-team.ru/Users/lagrunge/CGI>`_.
    Ответственные: mvel@.
    """
    type = 'ANNOTATE_CGI_PARAMS'
    input_parameters = (
        ArcadiaUrlParameter,
    )

    client_tags = ctc.Tag.Group.LINUX

    def on_execute(self):
        branch_path = self.ctx.get(ArcadiaUrlParameter.name)
        branch_name = branch_name_from_path(branch_path)

        os.chdir(self.abs_path())

        # branch should be proper here
        Arcadia.export(Arcadia.trunk_url('/search/tools/cgidocgen/cgidocgen.py'), self.abs_path('cgigen.py'))
        Arcadia.export(Arcadia.trunk_url('/search/tools/cgidocgen/mailer.py'), self.abs_path('mailer.py'))

        consumers_folder = "/search/request/consumers"
        consumers_path = branch_path + consumers_folder

        Arcadia.checkout(consumers_path, self.abs_path('consumers'))

        consumers = []

        def blame(svn_url, output_file):
            with open(self.abs_path(output_file), 'w') as bf:
                bf.write(Arcadia.blame(self.abs_path(svn_url), True))

        for root, dirs, files in os.walk(self.abs_path('consumers')):
            for file in files:
                if "consumer" in file and file.endswith(".cpp"):
                    consumers.append(os.path.join(root, file))

        for consumer in consumers:
            blame(self.abs_path(consumer), self.abs_path(consumer) + ".blame")

        run_process(["cat"] + [self.abs_path(c) for c in consumers] + [">>", self.abs_path("consumers/allconsumers.cpp")], shell=True)
        run_process(["cat"] + [self.abs_path(c + ".blame") for c in consumers] + [">>",  self.abs_path("consumers/allconsumers.cpp.blame")], shell=True)

        hsgenfile = branch_path + "/search/daemons/httpsearch/hsgenpages.cpp"
        Arcadia.export(hsgenfile, self.abs_path('hsgenpages.cpp'))
        blame(hsgenfile, 'hsgenpages.txt')
        token = self.get_vault_data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
        if run_process(
            "/skynet/python/bin/python {script}"
            " --cpp_file={cpp_file}"
            " --blame_from={blame_from}"
            " --wiki_path={wiki_path}"
            " --branch={branch}"
            " --blame_actions={blame_actions}"
            " --action_file={action_file}"
            " --oauth_token={token}".format(
                script=self.abs_path('cgigen.py'),
                cpp_file=self.abs_path('consumers/allconsumers.cpp'),
                blame_from=self.abs_path('consumers/allconsumers.cpp.blame'),
                wiki_path="JandeksPoisk/KachestvoPoiska/basesearch/cgi-auto/",
                branch=branch_name,
                blame_actions="hsgenpages.txt",
                action_file=self.abs_path('hsgenpages.cpp'),
                token="$TOKEN",
            ),
            log_prefix='annotater',
            shell=True,
            environment={
                "TOKEN": token,
            },
        ).returncode != 0:
            raise SandboxTaskFailureError('CGI Annotate Failed')


__Task__ = AnnotateCGI
