# -*- coding: utf-8 -*-

import os
import logging
import datetime
import shutil
import locale
import traceback
import itertools

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import add_write_permissions_for_path, make_folder, copytree3
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.errors import SandboxTaskUnknownError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.arcadia import sdk

from sandbox.sandboxsdk import parameters


DEFAULT_MR_USER = 'antirobot'
SCRIPTS_ARC = 'antirobot/scripts/learn'
SCHEME = '/serp_images'
FACTORS_VERSIONS = 'arcadia/antirobot/daemon_lib/factors_versions'
FEATURES_PURE_NAME = 'features_pure.txt'
FEATURES_TWEAKED_NAME = 'features_tweaked.txt'
FEATURES_TWEAKED_LEARN_NAME = 'features_tweaked_learn'
FEATURES_TWEAKED_TEST_NAME = 'features_tweaked_test'
TWEAK_SEPARATED_NAME = 'tweak_separated'
TWEAK_RESULT_LOG = 'tweak_log'

SCHEME_DICT = {'Random captcha': '/random_captchas', 'SERP counter': '/serp_images'}
ZONE_DICT = ((u'ВСЁ', 'all'), (u'КУБР', 'cubr'), (u'Турция', 'tr'), (u'КУБР + .com', 'cubr_com'))

TWEAK_NAMES = (
    'hidden_image',
    'req_entropy',
    'spravka_counts',
    'cookie_req_count',
    'hiload',
    'clicks',
    'patterns',
    'frauds',
    'req_login',
    'botnets',
    'bad_subnets',
    'too_many_redirects',
    'suspicious',
    'exp_random_captcha',
    'high_rps',
)

CtxTweakName = lambda n: 'use_tweak_%s' % n
CtxTweakLevel = lambda n: 'tweak_level_%s' % n

PythonExecutable = '/usr/bin/env python'


def checkDateYmd(value):
    try:
        datetime.datetime.strptime(value, '%Y%m%d')
        return True
    except:
        return False


def CopyTree(src, dst):
    names = os.listdir(src)

    os.makedirs(dst)
    for name in names:
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        if os.path.isdir(srcname):
            CopyTree(srcname, dstname)
        else:
            shutil.copy(srcname, dstname)


class AntirobotCollectFeatures(SandboxTask):
    type = 'ANTIROBOT_COLLECT_FEATURES'

    class ArcadiaSvnURL(parameters.SandboxArcadiaUrlParameter):
        name = consts.ARCADIA_URL_KEY
        description = 'Svn url for arcadia'
        default_value = Arcadia.trunk_url()

        @classmethod
        def cast(cls, value):
            if not value:
                return None if cls.required else ''
            if value.rfind('@') > 0 and value.rfind('/') < value.rfind('@'):
                value = value[:value.rfind('@')]
            value = value.rstrip('/')
            # Ensure sub-path "/devtools" or "/web" exists
            # (copied from projects.common.parameters.checker.check_build_svn_url)
            try:
                super(AntirobotCollectFeatures.ArcadiaSvnURL, cls).cast(value + '/devtools')
            except (ValueError, TypeError):
                super(AntirobotCollectFeatures.ArcadiaSvnURL, cls).cast(value + '/web')
            return super(AntirobotCollectFeatures.ArcadiaSvnURL, cls).cast(value)

    class DateBase(parameters.SandboxStringParameter):
        @classmethod
        def cast(cls, value):
            datetime.datetime.strptime(value, '%Y%m%d')
            return value

    class StartDate(DateBase):
        name = 'start_date'
        description = 'Start date (YYYYMMDD)'
        default_value = (datetime.datetime.today() - datetime.timedelta(days=31)).strftime("%Y%m%d")

    class EndDate(DateBase):
        name = 'end_date'
        description = 'End date (YYYYMMDD)'
        default_value = (datetime.datetime.today() - datetime.timedelta(days=1)).strftime("%Y%m%d")

    class TrainingSetGenScheme(parameters.SandboxSelectParameter):
        name = 'training_set_gen_scheme'
        description = 'Training set generation scheme'
        choices = SCHEME_DICT.items()

    class TweakLearnSet(parameters.SandboxBoolParameter):
        name = 'tweak_learn_set'
        description = 'Tweak learn set'
        default_value = True

    class MapreduceServer(parameters.SandboxStringParameter):
        name = 'mapreduce_server'
        description = 'Mapreduce server'
        default_value = 'sakura.search.yandex.net'

    class MaprduceUser(parameters.SandboxStringParameter):
        name = 'mapreduce_user'
        description = 'Mapreduce user'
        default_value = DEFAULT_MR_USER

    class Zone(parameters.SandboxSelectParameter):
        name = 'zone'
        description = u'Зона выборки'
        choices = ZONE_DICT

    class ResourcesBlock(parameters.SandboxInfoParameter):
        name = 'resources_info_field'
        description = 'Task resources'

    class MapreduceResourceId(parameters.LastReleasedResource):
        name = 'mapreduce_resource_id'
        description = 'Mapreduce executable'
        resource_type = resource_types.MAPREDUCE_EXECUTABLE

    class TweaksBlock(parameters.SandboxInfoParameter):
        name = 'tweaks_block'
        description = 'Tweaks'

    class ThresholdValue(parameters.SandboxFloatParameter):
        name = 'threshold_value'
        description = 'Tweak threshold'
        default_value = 30.0

    class AdditionalHacksBlock(parameters.SandboxInfoParameter):
        name = 'additional_hacks_block'
        description = 'Additional hacks'

    class HackMvideo(parameters.SandboxBoolParameter):
        name = 'hack_mvideo'
        description = 'm.video.yandex.ru'
        default_value = True

    class HackXWapProfile(parameters.SandboxBoolParameter):
        name = 'hack_x_wap_profile'
        description = 'x-wap-profile'
        default_value = True

    def make_tweak_parameters():
        '''('save_make_paths', field.BooleanFieldWithSubfields, 'Save make paths', {'default_value': False, 'sub_fields' : { 'true' : ['save_make_paths_attr']}}),
        '''
        return list(itertools.chain.from_iterable(
            [
                type(
                    '%s' % CtxTweakName(t),
                    (parameters.SandboxBoolParameter,),
                    dict(
                        name=CtxTweakName(t),
                        description='Use tweak %s' % t,
                        sub_fields={'true': [CtxTweakLevel(t)]}
                    )
                ),
                type(
                    '%s' % CtxTweakLevel(t),
                    (parameters.SandboxFloatParameter,),
                    dict(
                        name='%s' % CtxTweakLevel(t),
                        description='     Level of %s' % t,
                        default_value=10
                    )
                )
            ]
            for t in TWEAK_NAMES
        ))

    input_parameters = [
        ArcadiaSvnURL,
        StartDate,
        EndDate,
        TrainingSetGenScheme,
        TweakLearnSet,
        MapreduceServer,
        MaprduceUser,
        Zone,
        ResourcesBlock,
        MapreduceResourceId,
        TweaksBlock,
        ThresholdValue,
    ] + make_tweak_parameters() + [
        AdditionalHacksBlock,
        HackMvideo,
        HackXWapProfile,
    ]

    TARGET_RESOURCES = [
        ('features_pure_resource_id', FEATURES_PURE_NAME),
        ('features_tweaked_learn_res_id', FEATURES_TWEAKED_LEARN_NAME),
        ('features_tweaked_test_res_id', FEATURES_TWEAKED_TEST_NAME),
        ('rnd_data_resource_id', 'rnd_data'),
        ('tweak_log_res_id', 'tweak_log'),
        ('tweak_prepared_resource_id', 'tweak_prepared'),
        ('tweak_separated_resource_id', TWEAK_SEPARATED_NAME),
        ('untweak_log_res_id', 'untweak_log'),
        ]

    def initCtx(self):
        self.ctx['notify_via'] = 'email'
        self.ctx['execNumber'] = 0
        for r in self.TARGET_RESOURCES:
            self.ctx[r[0]] = None

    def update_path(self, path, depth='files'):
        dirs = path.split('/')
        if len(dirs) == 0:
            return

        cwd = os.getcwd()
        for folder in dirs[:-1]:
            Arcadia.update(folder, depth='empty')
            os.chdir(folder)
        Arcadia.update(dirs[-1], depth=depth)
        os.chdir(cwd)

    def DateIntervalDescr(self):
        return '%s-%s' % (self.ctx['start_date'], self.ctx['end_date'])

    def GetMapreducePath(self):
        return self.sync_resource(self.ctx['mapreduce_resource_id'])

    def MakeDateInterval(self):
        if not self.ctx.get('start_date'):
            return ''
        else:
            return ' %s-%s' % (self.ctx['start_date'], self.ctx['end_date'])

    def UpdateArcadia(self):
        self.arcadiaDir = os.path.join(self.abs_path(), 'arcadia')
        self.cachedArcDir = Arcadia.get_arcadia_src_dir(self.ctx[consts.ARCADIA_URL_KEY])

        # Copy arcadia subset into local task dir to get import hooks worked correctly
        def Cp(arcPath):
            copytree3('%s/%s' % (self.cachedArcDir, arcPath), '%s/%s' % (self.arcadiaDir, arcPath))

        Cp('contrib/libs')
        Cp('antirobot')
        Cp('devtools')
        Cp('yweb/scripts/datascripts/common')
        Cp('library/cpp/eventlog')

    def BuildProtobufs(self):
        # To successfule build protos, the following directory tree must be created:
        # taskDir
        #   + arcadia
        #   + build
        #       + release
        #
        buildType = 'release'
        self.buildDir = os.path.join(self.abs_path(), 'build', buildType)
        make_folder(self.buildDir)
        sdk.do_build(
             consts.YMAKE_BUILD_SYSTEM, self.cachedArcDir, ['antirobot/scripts/antirobot_eventlog'],
             buildType, clear_build=True, results_dir=self.buildDir)

    def RemoveResourceFilesIfAny(self, resName):
        resId = self.ctx.get(resName)
        if resId:
            self.remove_resource_files(resId)
            channel.sandbox.delete_resource(resId)
            self.ctx[resName] = None
            channel.sandbox.set_task_context_value(self.id, resName, None)

    def MakeRndData(self):
        def CopyToFactors(rndDataDir):
            shutil.copyfile('%s/factors_good.inc' % rndDataDir, '%s/factors_good.inc' % FACTORS_VERSIONS)

        logging.info('Making rnd_data...')
        self.set_info('Making rnd_data')

        os.chdir(self.abs_path())
        resId = self.ctx.get('rnd_data_resource_id')
        logging.info('Rnd data resource: %s' % str(resId))

        if resId:
            logging.info('rnd_data has been already prepared')
            logging.info('Try to sync rnd_data resource')

            self.sync_resource(resId)
            resObj = channel.sandbox.get_resource(resId)
            logging.info('Resource state: %s' % resObj.status)
            logging.info('Resource host: %s' % resObj.host)
            logging.info('Resource complete: %s' % resObj.complete)

            self.mark_resource_ready(resId)

            CopyToFactors(resObj.path)

            return resObj.path

        logging.info('There is no rnd_data resource - creating new')
        rndDataDir = 'rnd_data'

        os.chdir(self.abs_path())
        if not os.path.exists(rndDataDir):
            os.mkdir(rndDataDir)

        self.set_info('Executing extract_rnd_captcha_b')
        self._subprocess('%(python_executable)s %(scripts)s/extract_eventlog_data.py -m %(mapreducePath)s -r rnd_reqdata_raw -s %(server)s -u %(mr_user)s -z %(zone)s %(startDate)s %(endDate)s' % {
                'python_executable': PythonExecutable,
                'scripts': self.SCRIPTS + SCHEME,
                'mapreducePath': self.GetMapreducePath(),
                'server': self.ctx['mapreduce_server'],
                'startDate': self.ctx['start_date'],
                'endDate': self.ctx['end_date'],
                'mr_user': self.ctx['mapreduce_user'],
                'zone': self.ctx['zone'],
                },
                wait=True, check=True, out_file_name='%s/rnd_captchas' % rndDataDir, log_prefix='extract_rnd_captchas')

        self._subprocess('%s %s/tweak_factor_names.py %s' % (PythonExecutable, self.SCRIPTS, FACTORS_VERSIONS),
                wait=True, check=True, out_file_name='%s/factors_good.inc' % rndDataDir, log_prefix='tweak_factor_names')

        CopyToFactors(rndDataDir)

        self._subprocess('%s %s/convert_factors_to_one_version.py %s good' % (PythonExecutable,
                                                                              self.SCRIPTS + SCHEME,
                                                                              FACTORS_VERSIONS),
                wait=True, check=True, inp_file_name='%s/rnd_captchas' % rndDataDir, out_file_name='%s/rnd_captchas_oneversion' % rndDataDir, log_prefix='convert_factors')

        self._subprocess('%s %s/add_tweaking_factors.py -f %s/factors_good.inc -c %s/rnd_captchas_oneversion -r rnd_reqdata_raw' % (PythonExecutable,
                self.SCRIPTS + SCHEME, rndDataDir, rndDataDir),
                wait=True, check=True, out_file_name='%s/rnd_reqdata' % rndDataDir, log_prefix='add_valuable_factors')

        resObj = self.create_resource('(rnd_data) %s %s' % (self.descr, self.DateIntervalDescr()), rndDataDir, resource_types.ANTIROBOT_RND_DATA)
        self.mark_resource_ready(resObj)

        self.ctx['rnd_data_resource_id'] = resObj.id
        channel.sandbox.set_task_context_value(self.id, 'rnd_data_resource_id', resObj.id)

        return resObj.path

    def GetEnabledTweakNames(self):
        return [t for t in TWEAK_NAMES if self.ctx[CtxTweakName(t)]]

    def TweaksEnabled(self):
        return self.ctx['tweak_learn_set'] and self.GetEnabledTweakNames()

    def PrepareTweaks(self, rndDataPath):
        logging.info('Preparing tweaks...')
        self.set_info('Preparing tweaks')

        RESOURCE_PATH = 'tweak_prepared'

        def CreateResource(localPath):
            logging.info('Create resource for %s' % localPath)
            resObj = self.create_resource('(tweak_prepared) %s %s' % (self.descr, self.DateIntervalDescr()), localPath, resource_types.ANTIROBOT_TWEAK_PREPARED)
            self.mark_resource_ready(resObj)
            self.ctx['tweak_prepared_resource_id'] = resObj.id
            channel.sandbox.set_task_context_value(self.id, 'tweak_prepared_resource_id', self.ctx['tweak_prepared_resource_id'])

            return resObj.path

        curExecNumber = int(self.ctx['execNumber'])

        prevResourcePath = None
        newResourcePath = '%s/%s.%d' % (self.abs_path(), RESOURCE_PATH, curExecNumber)

        prevResId = self.ctx.get('tweak_prepared_resource_id')
        finished = self.ctx.get('tweak_prepared_finished')
        if prevResId:
            self.sync_resource(prevResId)
            resObj = channel.sandbox.get_resource(prevResId)
            self.mark_resource_ready(prevResId)
            prevResourcePath = resObj.path

            if finished:
                return prevResourcePath

            copytree3(prevResourcePath, os.path.join(self.abs_path(), newResourcePath))
            self.remove_resource_files(prevResId)

        if not os.path.exists(newResourcePath):
            os.mkdir(newResourcePath)

        add_write_permissions_for_path(newResourcePath)

        try:
            if self.TweaksEnabled():
                tweakList = self.GetEnabledTweakNames()

                self.set_info('Executing prepare step for tweaking learn set')
                self._subprocess('%(python_executable)s %(scripts)s/tweak/tweak_vyborka.py prepare --include=%(tweakList)s --rndreq=%(rndReq)s --mrexec=%(mrExec)s --server=%(server)s --prepared=%(preparedPath)s --proxy --mruser=%(mruser)s' % {
                        'python_executable': PythonExecutable,
                        'scripts': self.SCRIPTS,
                        'rndReq': '%s/rnd_reqdata' % rndDataPath,
                        'mrExec': self.GetMapreducePath(),
                        'server': self.ctx['mapreduce_server'],
                        'preparedPath': newResourcePath,
                        'mruser': self.ctx['mapreduce_user'],
                        'tweakList': ','.join(tweakList)
                    },
                    wait=True, check=True, log_prefix='tweak_vyborka_prepare')

            finished = True

            return newResourcePath
        finally:
            CreateResource(newResourcePath)
            self.ctx['tweak_prepared_finished'] = finished
            channel.sandbox.set_task_context_value(self.id, 'tweak_prepared_finished', finished)

    def MakePureFeatures(self, rndDataPath):
        logging.info('Making pure features...')
        self.set_info('Making pure features')

        resId = self.ctx.get('features_pure_resource_id')
        if resId:
            logging.info('Pure features already ready, syncing...')
            self.sync_resource(resId)
            resObj = channel.sandbox.get_resource(resId)
            self.mark_resource_ready(resId)
            return resObj.path

        resourcePath = FEATURES_PURE_NAME
        self._subprocess('%(python_executable)s %(scripts)s/make_matrixnet_features.py '\
                         '-b %(badSubnetsPath)s/bad_subnets -r %(rndDataPath)s/rnd_reqdata' % {
                    'python_executable': PythonExecutable,
                    'badSubnetsPath': self.SCRIPTS,
                    'scripts': self.SCRIPTS + SCHEME,
                    'rndDataPath': rndDataPath
                },
                wait=True, check=True, inp_file_name='%s/rnd_captchas_oneversion' % rndDataPath,
                out_file_name=resourcePath,
                log_prefix='make_features')

        resObj = self.create_resource('(pure features) %s %s' % (self.descr, self.DateIntervalDescr()), resourcePath, resource_types.ANTIROBOT_FEATURES)
        self.mark_resource_ready(resObj)
        self.ctx['features_pure_resource_id'] = resObj.id
        channel.sandbox.set_task_context_value(self.id, 'features_pure_resource_id', resObj.id)

        return resourcePath

    def MakeSeparatedTweaks(self, rndDataPath, preparedPath):
        resId = self.ctx.get('tweak_separated_resource_id')
        if resId:
            self.sync_resource(resId)
            resObj = channel.sandbox.get_resource(resId)
            self.mark_resource_ready(resId)
            return resObj.path

            logging.debug('MakeSeparatedTweaks(): preparedPath empty - quit')
            return None

        logging.info('Making separated tweaks')
        self.set_info('Making separated tweaks')

        resourcePath = os.path.join(self.abs_path(), TWEAK_SEPARATED_NAME)

        if not os.path.exists(resourcePath):
            os.mkdir(resourcePath)

        scripts = self.SCRIPTS
        resDescription = ''
        if self.TweaksEnabled() and preparedPath:
            # make single tweaks result
            python_executable = PythonExecutable
            self._subprocess('%(python_executable)s %(scripts)s/tweak/tweak_vyborka.py check --prepared=%(preparedPath)s --rndreq=%(rndDataPath)s/rnd_reqdata --tweak-dir=%(resourcePath)s' % locals(),
                wait=True, check=True, log_prefix='tweak_vyborka_check')
            resDescr = '(tweak separated) %s %s' % (self.descr, self.DateIntervalDescr())
        else:
            resDescr = 'Empty - tweaks were not created'

        resObj = self.create_resource('(tweak separated) %s %s' % (self.descr, self.DateIntervalDescr()), resourcePath, resource_types.ANTIROBOT_TWEAK_SEPARATED)
        self.mark_resource_ready(resObj)
        self.ctx['tweak_separated_resource_id'] = resObj.id
        channel.sandbox.set_task_context_value(self.id, 'tweak_separated_resource_id', self.ctx['tweak_separated_resource_id'])
        return resourcePath

    def PrepareLearnSets(self, features_path, tweak_separated_path, rndreq):
        logging.info('Preparing learn set...')
        self.set_info('Preparing learn set')

        os.chdir(self.abs_path())
        featuresTweaked = '%s/features_tweaked' % self.abs_path()

        def MakeLevels():
            levelsFileName = os.path.join(self.abs_path(), 'levels')
            if os.path.exists(levelsFileName):
                add_write_permissions_for_path(levelsFileName)

            f = open(levelsFileName, 'w')
            for t in TWEAK_NAMES:
                level = self.ctx.get(CtxTweakLevel(t))
                if level is not None:
                    print >>f, '%s=%f' % (t, level)
            f.close()
            return levelsFileName

        def ApplyTweaks():
            def StoreResource(filePath, description):
                res = self.create_resource(description, tweakLog, resource_types.ANTIROBOT_TWEAK_LOG)
                self.mark_resource_ready(res)
                self.ctx['tweak_log_res_id'] = res.id
                channel.sandbox.set_task_context_value(self.id, 'tweak_log_res_id', self.ctx['tweak_log_res_id'])

            self.RemoveResourceFilesIfAny('tweak_log_res_id')

            tweakLog = '%s/tweak_log' % self.abs_path()
            if not self.TweaksEnabled():
                self._subprocess('touch %s' % tweakLog, wait=True, check=True)
                StoreResource(tweakLog, "empty tweak log (tweaks were not applied)")
                logging.debug('ApplyTweaks(): tweak_separated_path empty - quit')
                return

            tweakList = self.GetEnabledTweakNames()

            levelsFileName = MakeLevels()
            self._subprocess('%(scripts)s/tweak/apply_tweaks.py apply --src-dir=%(tweak_dir)s --result=%(tweak_log)s --include=%(use_tweaks)s --threshold=%(threshold)d --tweak-levels=%(levelsFile)s' %
                {
                    'scripts': self.SCRIPTS,
                    'tweak_dir': tweak_separated_path,
                    'tweak_log': tweakLog,
                    'use_tweaks': ','.join(tweakList),
                    'threshold': self.ctx['threshold_value'],
                    'levelsFile': levelsFileName,
                }, wait=True, check=True, inp_file_name=features_path, out_file_name=featuresTweaked, log_prefix='apply_tweaks'
            )
            StoreResource(tweakLog, '(tweak log) %s%s' % (self.descr, self.MakeDateInterval()))

        # returns path to untweaked features
        def ApplyHacks(featuresPath):
            self.RemoveResourceFilesIfAny('untweak_log_res_id')

            untweakLog = 'untweak_log'
            res = featuresPath

            hackList = []
            if self.ctx['hack_mvideo']:
                hackList.append('m_video')

            if self.ctx['hack_x_wap_profile']:
                hackList.append('x_wap_profile')

            if hackList:
                self._subprocess(['%s/tweak/wrong_robots.py' % self.SCRIPTS, '--use=%s' % ','.join(hackList), '%s/rnd_reqdata' % rndreq, featuresPath],
                                    wait=True, shell=True, check=True, log_prefix='unset_wrong_robots_', out_file_name=untweakLog)

                featuresUntweaked = featuresPath + '.untweaked'
                self._subprocess('%s/tweak/apply_tweaks.py unapply --tweak-log=%s' % (self.SCRIPTS, untweakLog),
                    inp_file_name=featuresPath, out_file_name=featuresUntweaked, wait=True, check=True, log_prefix='unapply_tweaks_')

                res = featuresUntweaked
            else:
                self._subprocess('touch %s' % untweakLog, wait=True, check=True)

            untweakLogRes = self.create_resource('(untweak log) %s%s' % (self.descr, self.MakeDateInterval()), untweakLog, resource_types.ANTIROBOT_TWEAK_LOG)
            self.mark_resource_ready(untweakLogRes)
            self.ctx['untweak_log_res_id'] = untweakLogRes.id
            channel.sandbox.set_task_context_value(self.id, 'untweak_log_res_id', self.ctx['untweak_log_res_id'])

            return res

        featuresPath = features_path

        ApplyTweaks()
        featuresPath = ApplyHacks(featuresPath)

        featTweakedLearnPath = '%s/%s' % (self.abs_path(), FEATURES_TWEAKED_LEARN_NAME)
        featTweakedTestPath = '%s/%s' % (self.abs_path(), FEATURES_TWEAKED_TEST_NAME)

        self.RemoveResourceFilesIfAny('features_tweaked_learn_res_id')
        self.RemoveResourceFilesIfAny('features_tweaked_test_res_id')

        self._subprocess('%s/split_vyborka.py %s %s %s 0.9' % (self.SCRIPTS, featuresPath, featTweakedLearnPath, featTweakedTestPath),
                wait=True, check=True, log_prefix='split')

        res = self.create_resource('(tweaked features for learn) %s%s' % (self.descr, self.MakeDateInterval()), featTweakedLearnPath, resource_types.ANTIROBOT_FEATURES)
        self.mark_resource_ready(res)
        self.ctx['features_tweaked_learn_res_id'] = res.id
        channel.sandbox.set_task_context_value(self.id, 'untweak_log_res_id', self.ctx['features_tweaked_learn_res_id'])

        res = self.create_resource('(tweaked features for test) %s%s' % (self.descr, self.MakeDateInterval()), featTweakedTestPath, resource_types.ANTIROBOT_FEATURES)
        self.mark_resource_ready(res)
        self.ctx['features_tweaked_test_res_id'] = res.id
        channel.sandbox.set_task_context_value(self.id, 'features_tweaked_test_res_id', self.ctx['features_tweaked_test_res_id'])

    def on_execute(self):
        global SCHEME
        SCHEME = self.ctx['training_set_gen_scheme']

        self.ctx['execNumber'] = self.ctx['execNumber'] + 1
        channel.sandbox.set_task_context_value(self.id, 'execNumber', self.ctx['execNumber'])

        logging.info('Set locale as en_US.UTF-8')
        locale.setlocale(locale.LC_ALL, ('en_US', 'UTF-8'))
        logging.info('Current locale: %s' % str(locale.getlocale()))

        os.chdir(self.abs_path())

        self.UpdateArcadia()
        self.SCRIPTS = os.path.join(self.arcadiaDir, SCRIPTS_ARC)

        self.BuildProtobufs()

        try:
            rndDataPath = self.MakeRndData()
            pureFeaturesPath = self.MakePureFeatures(rndDataPath)

            separatedTweaksPath = None

            preparedPath = self.PrepareTweaks(rndDataPath)
            separatedTweaksPath = self.MakeSeparatedTweaks(rndDataPath, preparedPath)

            self.PrepareLearnSets(pureFeaturesPath, separatedTweaksPath, rndDataPath)

        except Exception:
            logging.info('Exception\n:%s' % traceback.format_exc())
            raise SandboxTaskUnknownError


__Task__ = AntirobotCollectFeatures
