import os
import errno
import logging
import json
import urlparse
import pathlib2
import subprocess

import requests
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.task as sdk_task
import sandbox.projects.common.search.performance as s_performance
import sandbox.projects.common.search.components.component as c_common
import sandbox.common.share as c_share
import sandbox.common.types.client as ctc

application_parameters_group = 'Application parameters'


class AppPerformanceTest(
    s_performance.OldShootingTask,
    sdk_task.SandboxTask,
):
    """ Application load testing """

    class CommandParameter(sdk_parameters.SandboxStringParameter):
        name = 'command_parameter'
        required = True
        description = 'Command with arguments. Use special words _IPORT_, _LOGS_DIR_. For example, ./suggest-web-daemon daemon.conf -p _IPORT_ -L _LOGS_DIR_/daemonlog'
        group = application_parameters_group

    class WaitURLParameter(sdk_parameters.SandboxStringParameter):
        name = 'wait_url_parameter'
        required = True
        description = 'Wait until daemon responds 200 to this url request. Use special word _IPORT_.'
        default_value = 'http://localhost:_IPORT_/ping'
        group = application_parameters_group

    class ResourcesParameter(sdk_parameters.ResourceSelector):
        name = 'resources_parameter'
        description = 'Sandbox resources. For example, binary, config and some indexes'
        multiple = True
        group = application_parameters_group

    class UrlFilesParameter(sdk_parameters.SandboxStringParameter):
        name = 'url_files_parameter'
        description = 'python dict with rbtorrents. For example, {"index": "rbtorrent:88...a", "index_fast": "rbtorrent:ad...82"}'
        group = application_parameters_group

    class NannyServiceParameter(sdk_parameters.SandboxStringParameter):
        name = 'nanny_service_parameter'
        description = 'Nanny service to get resources (without jinja files). For example, suggest_images_r1'
        group = application_parameters_group

    class PrepareCommandParameter(sdk_parameters.SandboxStringParameter):
        name = 'prepare_command_parameter'
        description = 'Prepare command. Use special words _IPORT_, _LOGS_DIR_. For example, echo "Hello, Kitty" > kitty._IPORT_.txt'
        group = application_parameters_group

    class DolbilkaPlanParameter(sdk_parameters.ResourceSelector):
        name = 'dolbilka_plan_parameter'
        required = True
        description = 'Dolbilka plan. You should upload queries file to sandbox as PLAIN_TEXT_QUERIES resource type and run GENERATE_PLAN_FROM_QUERIES task.'
        group = application_parameters_group

    class StartTimeoutParameter(sdk_parameters.SandboxIntegerParameter):
        name = 'start_timeout_parameter'
        description = 'Executable daemon start timeout in seconds.'
        default_value = 600
        group = application_parameters_group

    class ShutdownTimeoutParameter(sdk_parameters.SandboxIntegerParameter):
        name = 'shutdown_timeout_parameter'
        description = 'Executable daemon shutdown timeout in seconds.'
        default_value = 10
        group = application_parameters_group

    type = 'APP_PERFORMANCE_TEST'

    input_parameters = list(s_performance.OldShootingTask.shoot_input_parameters) + [
        CommandParameter,
        WaitURLParameter,
        ResourcesParameter,
        UrlFilesParameter,
        NannyServiceParameter,
        PrepareCommandParameter,
        DolbilkaPlanParameter,
        StartTimeoutParameter,
        ShutdownTimeoutParameter
    ]

    client_tags = ctc.Tag.Group.LINUX & ctc.Tag.LXC

    def replace_instance_vars(self, text):
        instance_vars = {
            '_IPORT_': str(self.ctx['instance_port']),
            '_LOGS_DIR_': self.log_path()
        }
        for var in instance_vars:
            text = text.replace(var, instance_vars[var])
        return text

    def on_execute(self):
        logging.info('PWD %s', os.getcwd())

        if self.ctx['nanny_service_parameter']:
            self.download_nanny_service_resources(self.ctx['nanny_service_parameter'])

        if self.ctx['resources_parameter']:
            for resource_id in self.ctx['resources_parameter']:
                self.download_sandbox_resource(resource_id)

        if self.ctx['url_files_parameter']:
            self.download_url_resources(self.ctx['url_files_parameter'])

        # ls current working directory
        cwd_files = ' '.join(os.listdir(os.curdir))
        logging.info('Files in current directory: %s', cwd_files)

        # Replace _PORT_ to instance port
        self.ctx['instance_port'] = c_common.try_get_free_port(tries=25)
        self.ctx['command_real'] = self.replace_instance_vars(self.ctx['command_parameter'])
        self.ctx['wait_url_real'] = self.replace_instance_vars(self.ctx['wait_url_parameter'])
        logging.info('Command line: %s. Wait URL: %s', self.ctx['command_real'], self.ctx['wait_url_real'])

        # Run prepare command
        if self.ctx['prepare_command_parameter']:
            self.ctx['prepare_command_real'] = self.replace_instance_vars(self.ctx['prepare_command_parameter'])
            self.prepare_process = sdk_process.run_process(
                self.ctx['prepare_command_real'],
                outputs_to_one_file=False,
                log_prefix="prepare command",
                wait=True,
                shell=True
            )

        my_daemon = c_common.SearchDaemonComponent(
            args=self.ctx['command_real'].split(' '),
            start_timeout=self.ctx['start_timeout_parameter'],
            shutdown_timeout=self.ctx['shutdown_timeout_parameter'],
            url=self.ctx['wait_url_real'],
        )
        my_daemon.port = self.ctx['instance_port']

        self._init_virtualenv()

        # Start daemon and launch load testing
        with my_daemon:
            self._old_shoot(
                my_daemon,
                self.ctx['dolbilka_plan_parameter'],
            )

    def download_sandbox_resource(self, resource_id, local_path=''):
        resource = requests.get('https://sandbox.yandex-team.ru:443/api/v1.0/resource/{}'.format(resource_id)).json()
        if not local_path:
            local_path = os.path.basename(resource['file_name'])
        self.sky_get_like_iss(resource['skynet_id'], local_path)

    def download_url_resources(self, urls):
        # Download rbtorrents
        # TODO: add possibility to download http(s):// resources
        urls_dict = json.loads(urls)
        for local_path, url in urls_dict.iteritems():
            self.sky_get_like_iss(url, local_path)

    def sky_files(self, rbtorrent):
        files = subprocess.check_output(['sky', 'files', '--json', rbtorrent])
        return json.loads(files)

    def sky_get_like_iss(self, url, destination):
        logging.info('Download rbtorrent %s into %s', url, destination)
        torrent_files = self.sky_files(url)
        # if rbtorrent has a directory then there is will be additional element in torrent_files.
        # So if len=1 then there is only one file without directory
        if len(torrent_files) == 1 and '/' not in torrent_files[0]['name']:
            c_share.skynet_get(url, '.')
            os.rename(torrent_files[0]['name'], destination)
        else:
            c_share.skynet_get(url, destination)

    def download_nanny_service_resources(self, service):
        NANNY_URL = 'http://nanny.yandex-team.ru/'
        OAUTH_TOKEN = self.get_vault_data('SUGGEST', 'robot-rcss-autoadmin-nanny-oauth')
        YAV_OAUTH_TOKEN = self.get_vault_data('SUGGEST', 'robot-rcss-autoadmin-yav-token')

        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(OAUTH_TOKEN)
        session.headers['Content-Type'] = 'application/json'

        nanny_config = session.get(urlparse.urljoin(NANNY_URL, 'v2/services/' + service + '/runtime_attrs/')).json()['content']
        resources = nanny_config['resources']
        logging.info('Resources from nanny service %s', json.dumps(resources))

        for volume in nanny_config['instance_spec']['volume']:
            if volume['type'] == 'VAULT_SECRET':
                secret_ver = volume['vaultSecretVolume']['vaultSecret']['secretVer']
                secret = (
                    requests.Session()
                    .get(
                        'https://vault-api.passport.yandex.net/1/versions/{}'.format(secret_ver),
                        headers={'Content-Type': 'application/json', 'Authorization': 'Oauth {}'.format(YAV_OAUTH_TOKEN)},
                        verify=False,
                    )
                    .json()
                )
                if (secret['status'] != 'ok'):
                    logging.info('Failed to obtain secret {}'.format(secret_ver))
                else:
                    for obj in secret['version']['value']:
                        pathlib2.Path(volume['name']).mkdir(parents=True, exist_ok=True)
                        local_path = os.path.join(volume['name'], obj['key'])
                        logging.info('Writing secret into %s', local_path)
                        with open(local_path, 'w') as f:
                            f.write(obj['value'].encode('utf-8'))
            else:
                logging.info('Unsupported volume type %s by name %s', volume['type'], volume['name'])

        for file in resources['sandbox_files']:
            logging.info('Sandbox file %s from %s', file['resource_type'], file['task_id'])
            resource_id = self.get_resource_id_from_task(file['task_id'], file['resource_type'])
            logging.info('Sandbox resource %s ', resource_id)
            self.download_sandbox_resource(resource_id, file['local_path'])

        for file in resources['url_files']:
            self.sky_get_like_iss(file['url'], file['local_path'])

        for file in resources['static_files']:
            with open(file['local_path'], 'w') as f:
                f.write(file['content'].encode("utf-8"))

    def force_symlink(self, src, dest):
        try:
            os.symlink(src, dest)
        except OSError, e:
            if e.errno == errno.EEXIST:
                os.remove(dest)
                os.symlink(src, dest)

    def get_resource_id_from_task(self, task_id, resource_type):
        resources_response = requests.get('https://sandbox.yandex-team.ru:443/api/v1.0/resource?task_id={}&type={}&limit=1'.format(task_id, resource_type))
        res = json.loads(resources_response.text)['items'][0]
        return res['id']


__Task__ = AppPerformanceTest
