# -*- coding: utf-8 -*-

__author__ = 'bykanov'

import copy

import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.utils as utils
from sandbox.projects.AppPerformanceTest import AppPerformanceTest

application_parameters_group = 'Supermind parameters'


class AppPerformanceTestBest(AppPerformanceTest):
    """ Start multiple AppPerformanceTest tasks and choose the best one """

    class NumberOfRunsParameter(sdk_parameters.SandboxIntegerParameter):
        name = 'number_of_runs'
        description = 'How many tests to execute'
        default_value = 3
        group = application_parameters_group

    class CpuModelParameter(sdk_parameters.SandboxStringParameter):
        name = 'cpu_model'
        description = 'Required CPU model for tests. For example, e5-2650'
        group = application_parameters_group

    class RamParameter(sdk_parameters.SandboxIntegerParameter):
        name = 'ram'
        description = 'Required RAM(GB) for tests. For example, 10'
        group = application_parameters_group
        required = True
        default_value = 2

    class DiskSpaceParameter(sdk_parameters.SandboxFloatParameter):
        name = 'disk_space'
        description = 'Required disk space(GB) for tests. For example, 50'
        group = application_parameters_group
        required = True
        default_value = 20

    type = 'APP_PERFORMANCE_TEST_BEST'

    input_parameters = list(AppPerformanceTest.input_parameters) + [
        NumberOfRunsParameter, CpuModelParameter, RamParameter, DiskSpaceParameter
    ]

    def on_execute(self):
        number_of_runs = self.ctx['number_of_runs']
        subtasks = self.list_subtasks(load=True)
        if not subtasks:
            for i in range(int(number_of_runs)):
                sub_ctx = copy.deepcopy(self.ctx)
                sub_ctx['notify_via'] = ''
                sub_ctx['notify_if_failed'] = self.owner
                if self.ctx['cpu_model']:
                    subtasks_model = self.ctx['cpu_model']
                if self.ctx['ram']:
                    subtasks_ram = self.ctx['ram']
                if self.ctx['disk_space']:
                    subtasks_execution_space = self.ctx['disk_space']
                subtasks_model = self.ctx['cpu_model'] if self.ctx['cpu_model'] else ''
                subtasks_ram = self.ctx['ram'] * 1024 if self.ctx['ram'] else ''
                subtasks_execution_space = self.ctx['disk_space'] * 1024 if self.ctx['disk_space'] else ''
                self.create_subtask(
                    task_type=AppPerformanceTest.type,
                    input_parameters=sub_ctx,
                    description="{} #{}".format(self.descr, i),
                    model=subtasks_model,
                    ram=subtasks_ram,
                    execution_space=subtasks_execution_space,
                )
            utils.wait_all_subtasks_stop()
        elif utils.check_all_subtasks_done():
            # See https://ml.yandex-team.ru/thread/2370000002596347627/
            completed_count = len([subtask for subtask in subtasks if subtask.is_completed()])
            eh.verify(
                completed_count == number_of_runs,
                'Incorrect number of completed child tasks, {} != {}'.format(completed_count, number_of_runs)
            )
            utils.check_subtasks_fails(fail_on_failed_children=False)
        else:
            utils.restart_broken_subtasks()

        # even if children are not finished yet, we still can collect stats,
        # this also covers stupid race conditions like multiple DRAFTs of single run
        self._collect_reqs_stats(subtasks)

    def _collect_reqs_stats(self, subtasks):
        for ctx_field in ['requests_per_sec', 'memory_rss', 'memory_vsz', 'notfound_rates', 'results', 'fail_rates']:
            self.ctx[ctx_field] = []
            for s in subtasks:
                if ctx_field in s.ctx:
                    self.ctx[ctx_field] += s.ctx[ctx_field]
        self.ctx['max_rps'] = max(self.ctx['requests_per_sec'])


__Task__ = AppPerformanceTestBest
