import logging
import tempfile
import uuid
import re
import os

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.environments as environments
import sandbox.sandboxsdk.process as process
import sandbox.sandboxsdk.paths as paths
from sandbox.sandboxsdk.environments import VirtualEnvironment, PipEnvironment

import sandbox.projects.common.build.ArcadiaTask as arcadia_task
from sandbox.projects.common.vcs import arc


VAULT_PATTERN = r'\$\(vault:(?P<owner>[^:]+):(?P<name>[^:]+)\)'


class PathToScriptParam(parameters.SandboxStringParameter):
    name = 'path_to_script'
    description = 'Path to script'
    default_value = ''


class ExtraReqParam(parameters.SandboxStringParameter):
    name = 'extra_reqs'
    description = 'Extra requirements'
    default_value = ''


class ArgsParam(parameters.SandboxStringParameter):
    name = 'script_args'
    description = 'Args to script'
    default_value = ''


class ScriptParam(parameters.SandboxStringParameter):
    name = 'script'
    description = 'Post Execution Script'
    default_value = ''
    multiline = True


class DNS64Param(parameters.SandboxBoolParameter):
    name = 'dns64'
    description = 'Use DNS64'


class ArcadiaPyScriptRunner(arcadia_task.ArcadiaTask):
    type = "ARCADIA_PY_SCRIPT_RUNNER"

    environment = [environments.SvnEnvironment()]
    client_tags = ctc.Tag.GENERIC | ctc.Tag.Group.OSX
    cores = 17
    input_parameters = [
        parameters.SandboxArcadiaUrlParameter,
        PathToScriptParam,
        ArgsParam,
        ScriptParam,
        ExtraReqParam,
        DNS64Param,
    ]

    def __init__(self, *args, **kwargs):
        arcadia_task.ArcadiaTask.__init__(self, *args, **kwargs)
        self._abs_script_path = None

    @property
    def dns(self):
        return ctm.DnsType.DNS64 if self.ctx.get(DNS64Param.name) else ctm.DnsType.DEFAULT

    def deref(self, s):

        deref_dir = tempfile.mkdtemp()

        def deref_vault(match):
            deref_path = os.path.join(deref_dir, str(uuid.uuid4()))

            with open(deref_path, 'w') as f:
                f.write(self.get_vault_data(match.group('owner'), match.group('name')))

            return deref_path

        s = re.sub(VAULT_PATTERN, deref_vault, s)

        return s

    def on_execute(self):
        def mine_reqs(dir):
            req_file = os.path.join(dir, 'requirements.txt')
            if os.path.exists(req_file):
                return open(req_file, 'r').read().splitlines()
            return []

        def split_version(str):
            if '==' in str:
                return str.split('==', 1)
            return str, None

        cwd = self.log_path('cwd')
        paths.make_folder(cwd)
        os.chdir(cwd)

        with VirtualEnvironment() as venv:
            reqs = []
            with arc.Arc().mount_path(None, None, fetch_all=False) as arc_dir:
                if self.ctx['path_to_script']:
                    self._abs_script_path = os.path.join(arc_dir, self.ctx['path_to_script'])
                    reqs += mine_reqs(os.path.dirname(self._abs_script_path))

                reqs += self.ctx['extra_reqs'].split()
                reqs = map(split_version, reqs)

                logging.debug('Extra requirements %s', reqs)

                for req, version in reqs:
                    PipEnvironment(req, version, venv=venv).prepare()

                if self._abs_script_path:
                    process.run_process(
                        [venv.executable, self._abs_script_path] + self.deref(self.ctx['script_args']).split(),
                        log_prefix='script',
                        outputs_to_one_file=False,
                        shell=True,
                        work_dir=cwd,
                    )

        if self.ctx['script']:
            exec self.ctx['script'] in globals(), locals()


__Task__ = ArcadiaPyScriptRunner
