import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.vcs import arc
from sandbox.projects.common.YMakeBase import YmakeWrapper
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process, check_process_timeout
from subprocess import call
import os.path
import json
import re
import itertools
import datetime
import logging


class AutocheckCMakeListsUnreachability(ArcadiaTask):
    type = 'AUTOCHECK_CMAKELISTS_UNREACHABILITY'
    SANDBOX_TASKS_OWNER = 'g:sandbox-dev'
    client_tags = ctc.Tag.LINUX_PRECISE
    cores = 1
    required_ram = 32 * 1024
    execution_space = 32 * 1024

    def __init__(self, task_id=0):
        ArcadiaTask.__init__(self, task_id)

    def initCtx(self):
        self.execution_space = 10000  # 10Gb
        self.ctx['notify_via'] = 'email'

    def calc_all_intersections(self, platform_files):
        result = {}
        for i in reversed(range(len(platform_files))):
            combs = list(itertools.combinations(platform_files.keys(), i + 1))
            # combs like [('free', 'linux'), ('free', 'win'), ('linux', 'win')]
            for comb in combs:
                is_first = True
                for platform_name in comb:
                    if is_first:
                        intersection = set(platform_files[platform_name])
                        is_first = False
                    else:
                        intersection = intersection & platform_files[platform_name]
                result['+'.join(comb)] = list(intersection)
                for platform_name in comb:
                    platform_files[platform_name] -= intersection
        return result

    def write_json_file(self, file_name, data):
        file_path = file_name if os.path.isabs(file_name) else self.result_path(file_name)
        with open(file_path, "w") as fp:
            json.dump(data, fp, indent=2)

    def load_json_file(self, file_path):
        with open(file_path, "r") as fp:
            return json.load(fp)

    def write_log_file(self, file_name, data):
        res = ''
        for line in data:
            res += line + '\n'
        open(os.path.join(self.result_dir, file_name), "w").write(res)

    def calculate_cmakelists_freshness(self, result_without_owners, last_cmakelists_freshness):
        today = datetime.datetime.today().date()
        new_cmakelists_freshness = {'date': str(today)}

        delta = 0
        if last_cmakelists_freshness:
            delta = today - datetime.datetime.strptime(last_cmakelists_freshness['date'], '%Y-%m-%d').date()
        logging.info('Date delta {}'.format(delta))

        for lst in result_without_owners:
            if last_cmakelists_freshness and lst in last_cmakelists_freshness:
                new_cmakelists_freshness[lst] = last_cmakelists_freshness[lst] + delta.days
            else:
                new_cmakelists_freshness[lst] = 0

        file_name = 'cmakelists_freshness.json'

        self.write_json_file(file_name, new_cmakelists_freshness)
        self.write_json_file(self.abs_path(file_name), new_cmakelists_freshness)
        resource = self._create_resource(
            "AutocheckYMakeListsUnreachability freshness",
            os.path.join(self.abs_path(file_name)),
            resource_types.AUTOCHECK_CMAKELISTS_UNREACHABILITY_FRESHNESS
        )
        resource.mark_ready()

    def process(self, found_YMakeLists, well_known_cases_file, dump_results, last_cmakelists_freshness):

        with open(found_YMakeLists, 'r') as f:
            all_dirs = f.read().splitlines()
        statistics = ["All dirs count - {0}".format(str(len(all_dirs)))]

        errors = {'all_unreachable': []}
        well_known_dirs = open(well_known_cases_file, 'r').read().splitlines()
        platforms_reachabale_dirs = set()
        platform_files = {}
        for k in dump_results.keys():
            logging.info('Loading {} ...'.format(dump_results[k]))
            with open(dump_results[k], 'r') as fp:
                files = set(fp.read().splitlines())
                platform_files[k] = files
                if len(files) == 0:
                    errors['all_unreachable'] += [k]
                platforms_reachabale_dirs |= files
                statistics.append("{0} reachabale dirs count = {1}".format(k, str(len(files))))

        all_intersections = self.calc_all_intersections(platform_files)
        self.write_json_file('all_intersections.json', all_intersections)

        reachable_dirs_count = len(platforms_reachabale_dirs)
        raw_result = list(set(all_dirs) - platforms_reachabale_dirs)
        statistics.append("Unreachabale dirs count = {0}".format(str(len(raw_result))))
        result_without_owners = []
        filtered_dirs = {}
        for x in raw_result:
            matched_obj = re.match('|'.join(well_known_dirs), x)
            if matched_obj:
                if matched_obj.group(0) not in filtered_dirs:
                    filtered_dirs[matched_obj.group(0)] = {}
                else:
                    filtered_dirs[matched_obj.group(0)][x] = []
            else:
                result_without_owners.append(x)
        statistics.append(
            "Unreachabale dirs count after subtraction well known dirs = {0}".format(
                str(len(result_without_owners))
            )
        )

        self.calculate_cmakelists_freshness(result_without_owners, last_cmakelists_freshness)

        result_without_owners_full_path = [os.path.join(self.arcadia_src_dir, x.rstrip()) for x in result_without_owners]
        lists2owners = self.get_owners(result_without_owners_full_path)

        project_owners = []
        for projects in filtered_dirs.values():
            for proj in projects.keys():
                project_owners.append(os.path.join(self.arcadia_src_dir, proj.rstrip()))

        projects = self.get_owners(project_owners)

        owners2lists = {}
        for clist_dir, owners in lists2owners.iteritems():
            clist = os.path.join(clist_dir, 'ya.make')
            if clist.startswith('sandbox/projects'):
                if self.SANDBOX_TASKS_OWNER not in owners2lists:
                    owners2lists[self.SANDBOX_TASKS_OWNER] = []
                owners2lists[self.SANDBOX_TASKS_OWNER].append(clist)
                continue

            if not owners:
                if '' not in owners2lists:
                    owners2lists[''] = []
                owners2lists[''].append(clist)
            else:
                for owner in owners:
                    if owner in owners2lists:
                        owners2lists[owner].append(clist)
                    else:
                        owners2lists[owner] = [clist]

        # self.write_log_file('_well_known_dirs.txt', well_known_dirs)
        self.write_log_file('_result_without_filtration.txt', sorted(raw_result))
        self.write_log_file('_result_without_owners.txt', sorted(result_without_owners))
        self.write_json_file('Result.json', owners2lists)
        self.write_json_file('Errors.json', errors)
        self.write_json_file('filtered_lists.json', filtered_dirs)
        self.write_log_file('Statistics.txt', sorted(statistics))

        result_for_dash = {
            'projects': sum(len(lists) for lists in owners2lists.values()),
            'owners': len(owners2lists)
        }
        self.write_json_file('result_for_dashboard.json', result_for_dash)

        resource = self._create_resource(
            "AutocheckYMakeListsUnreachability result",
            self.result_dir,
            resource_types.AUTOCHECK_CMAKELISTS_UNREACHABILITY_RESULT
        )
        resource.mark_ready()
        self.ctx['unreachable_dirs_count'] = str(len(result_without_owners))
        self.ctx['reachable_dirs_count'] = str(reachable_dirs_count)
        self.ctx['resource_id'] = resource.id
        self.ctx['resource_url'] = resource.http_url()

    def run_cmd(self, cmd, cmdCtx):
        self.ctx[cmdCtx.logName + '_cmd'] = cmd
        logFile = self.result_path(cmdCtx.logName + cmdCtx.outputExt)
        errLogFile = self.result_path(cmdCtx.logName + cmdCtx.errOutputExt)

        with open(logFile, 'w') as output:
            with open(errLogFile, 'w') as errOutput:
                try:
                    cmdCtx.Verify()
                    p = run_process(
                        cmd, log_prefix=cmdCtx.logName, wait=cmdCtx.wait,
                        environment=cmdCtx.env, work_dir=cmdCtx.workDir,
                        stdin=cmdCtx.stdin, stdout=output, stderr=errOutput,
                        outputs_to_one_file=False
                    )
                except Exception as exc:
                    raise SandboxTaskFailureError(exc)
        return p, logFile

    def run_ymake(self, ymakeEx):
        return self.run_cmd(ymakeEx.ConstructCmd(), ymakeEx.cmdCtx)

    def result_path(self, p=""):
        return os.path.join(self.result_dir, p) if p else self.result_dir

    def get_owners(self, paths):
        ya_project_owners = YmakeWrapper(srcRoot=self.arcadia_src_dir, logName='ya_project_owners', subcommands=['project', 'owner'], use_autocheck_defines=False)
        ya_project_owners.args = ['--json-output'] + paths
        ya_project_owners.cmdCtx.workDir = self.arcadia_src_dir
        ya_project_owners.cmdCtx.outputExt = '.json'
        process, dump_file = self.run_ymake(ya_project_owners)
        check_process_timeout(process, 900, timeout_sleep=5)
        with open(dump_file) as dump:
            return json.load(dump)

    def get_rbgroups_full_data(self):
        ya_dump_groups = YmakeWrapper(srcRoot=self.arcadia_src_dir, logName='ya_dump_groups', subcommands=['dump', 'groups'], use_autocheck_defines=False)
        ya_dump_groups.cmdCtx.workDir = self.arcadia_src_dir
        ya_dump_groups.cmdCtx.outputExt = '.json'
        process, dump_file = self.run_ymake(ya_dump_groups)
        check_process_timeout(process, 900, timeout_sleep=5)
        with open(dump_file) as dump:
            return json.load(dump)

    def do_execute(self):
        with arc.Arc().mount_path(None, None, fetch_all=False) as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir
            self.result_dir = self.abs_path('result')
            make_folder(self.result_dir)

            rbgroups_full_data = self.get_rbgroups_full_data()

            last_res = apihelpers.get_last_resource(resource_types.AUTOCHECK_CMAKELISTS_UNREACHABILITY_FRESHNESS)
            last_cmakelists_freshness = None
            if last_res:
                last_cmakelists_path = self.sync_resource(last_res)
                logging.info("Path to old cmakelists freshness file %s", last_cmakelists_path)
                last_cmakelists_freshness = self.load_json_file(last_cmakelists_path)

            self.write_json_file('_rbgroups_full_data.json', rbgroups_full_data)

            current_dir = os.path.join(
                self.arcadia_src_dir,
                'sandbox', 'projects', 'AutocheckCMakeListsUnreachability'
            )
            well_known_cases_file = os.path.join(current_dir, 'WellKnownCases.txt')

            found_YMakeLists = os.path.join(self.result_dir, '_found_YMakeLists.txt')
            script_path = os.path.join(current_dir, 'findAllYMakeLists.sh')
            call([script_path, self.arcadia_src_dir, found_YMakeLists])

            ya = YmakeWrapper(srcRoot=self.arcadia_src_dir, logName='ya_dump', subcommands=['dump', 'dir-graph'])
            ya.args = ['--reachable-dirs', '--force-build-depends']
            ya.cmdCtx.workDir = self.arcadia_src_dir
            ya.cmdCtx.outputExt = '.txt'
            dump_results = {}
            platform_toolchains = {
                'freebsd': {'host': 'DEFAULT-FREEBSD-X86_64'},
                'linux': {'host': 'DEFAULT-LINUX-X86_64'},
                'windows': {'host': 'DEFAULT-WIN-X86_64'},
                'darwin': {'host': 'DEFAULT-DARWIN-X86_64'},
                'ios': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-IOS-X86_64'},
                'ios_maps': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-IOS-X86_64', 'target_flags': ['MAPSMOBI_BUILD_TARGET=yes']},
                'android': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-ANDROID-ARMV7A'},
                'jdk8': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-LINUX-X86_64', 'target_flags': ['JDK_VERSION=8']},
                'jdk11': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-LINUX-X86_64', 'target_flags': ['JDK_VERSION=11']},
                'jdk15': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-LINUX-X86_64', 'target_flags': ['JDK_VERSION=15']},
                'musl': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-LINUX-X86_64', 'target_flags': ['MUSL']},
                'asan': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-LINUX-X86_64', 'target_flags': ['SANITIZER_TYPE=address']},
                'msan': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-LINUX-X86_64', 'target_flags': ['SANITIZER_TYPE=memory']},
                'pgsan': {'host': 'DEFAULT-LINUX-X86_64', 'target': 'DEFAULT-LINUX-X86_64', 'target_flags': ['SANITIZER_TYPE=pg']},
            }
            for platform, toolchain in platform_toolchains.iteritems():
                ya_for_platform = ya.Copy()
                if 'host' in toolchain:
                    ya_for_platform.args += ['--host-platform={}'.format(toolchain['host'])]
                if 'target' in toolchain:
                    ya_for_platform.args += ['--target-platform={}'.format(toolchain['target'])]
                if 'target_flags' in toolchain:
                    target_flags = []
                    for flag in toolchain['target_flags']:
                        target_flags += ['--target-platform-flag', flag]
                    ya_for_platform.args += target_flags
                ya_for_platform.args += ['autocheck']
                ya_for_platform.cmdCtx.logName = 'ya_dump_{0}'.format(platform)
                process, dump_file = self.run_ymake(ya_for_platform)
                check_process_timeout(process, 21600, timeout_sleep=1)
                dump_results[platform] = dump_file
            try:
                self.process(found_YMakeLists, well_known_cases_file, dump_results, last_cmakelists_freshness)
            except Exception as e:
                resource = self._create_resource(
                    "AutocheckCoverage resource",
                    self.result_dir,
                    resource_types.AUTOCHECK_CMAKELISTS_UNREACHABILITY_RESULT
                )
                resource.mark_ready()
                raise e


__Task__ = AutocheckCMakeListsUnreachability
