# coding: utf-8

import os
import json
import logging
import textwrap
import shutil
import multiprocessing

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.import_tools import import_path


class __Task__(ArcadiaTask):
    type = 'AUTOCHECK_COMPARE_DEPENDENCIES'
    execution_space = 300000
    cores = 24
    client_tags = ctc.Tag.Group.LINUX

    def dump_ymake_deps(self, ya, source_root):
        ymake_build_root = self.abs_path('ymake_build_root')
        ydump_path = self.abs_path('ydump')
        yerr_path = self.abs_path('yerr')

        ymake_args = [
            ya,
            'tool',
            'ymake',
            '--',
        ]
        args = ymake_args + [
            '-n',
            '-k',
            '-xw',
            '-b', ymake_build_root,
            'release',
            source_root,
        ]

        with open(ydump_path, 'wb') as ydump, open(yerr_path, 'wb') as yerr:
            run_process(
                args,
                log_prefix='ymake_dump',
                stdout=ydump,
                stderr=yerr,
                wait=True,
                check=False,
                outputs_to_one_file=False,
            )

        return ydump_path

    def get_gcc_deps(self, ya, source_root):
        build_root = self.abs_path('ya_build_root')
        ya_build_stdout = self.abs_path('ya_build_stdout')
        ya_build_stderr = self.abs_path('ya_build_stderr')

        args = [
            ya,
            'build',
            '-j', str(multiprocessing.cpu_count()),
            '-k',
            '--build=release',
            '--source-root={}'.format(source_root),
            '--build-dir={}'.format(build_root),
            source_root,
        ]

        with open(ya_build_stdout, 'wb') as stdout, open(ya_build_stderr, 'wb') as stderr:
            run_process(
                args,
                log_prefix='ymake_dump',
                stdout=stdout,
                stderr=stderr,
                wait=True,
                check=False,
                outputs_to_one_file=False,
            )

        return build_root

    def on_execute(self):
        source_root = Arcadia.get_arcadia_src_dir(self.ctx[consts.ARCADIA_URL_KEY])
        svn_info = Arcadia.info(source_root)
        logging.debug("svn_info={svn_info}".format(svn_info=svn_info))
        revision = svn_info['commit_revision']
        self.ctx['revision'] = revision

        compare_dependencies = import_path(os.path.join(source_root, 'devtools', 'autocheck', 'compare_depends', 'compare_dependencies.py'))

        shutil.copyfile(
            os.path.join(source_root, 'devtools', 'autocheck', 'compare_depends', 'local.ymake'),
            os.path.join(source_root, 'local.ymake'),
        )

        ya = os.path.join(source_root, 'devtools', 'ya-dev', 'ya')
        ydump_path = self.dump_ymake_deps(ya, source_root)
        build_root = self.get_gcc_deps(ya, source_root)

        cache_path = self.abs_path('compare_cache.pickle')
        popularity_path = self.abs_path('missing_deps_by_popularity.txt')
        targets_path = self.abs_path('missing_deps_by_target.txt')
        c = compare_dependencies.Comparator(
            dump_file=ydump_path,
            build_root=os.path.join(build_root, 'latest'),
            source_root=source_root,
            cache_path=cache_path,
            popularity_path=popularity_path,
            targets_path=targets_path,
        )
        result = c.compare()
        c.print_result(result)

        result_path = self.abs_path('result.json')
        json.dump(result, open(result_path, 'wb'))

        # find previous result(s) before we create a new one
        old_results = channel.sandbox.list_resources(
            resource_type=resource_types.COMPILER_DEPENDENCIES_DIFF_RESULT,
            omit_failed=True,
            limit=1,
            order_by='-id',
        )
        logging.debug("old_results={}".format(old_results))

        # now creating resources for all the data we generated
        res = self.create_resource(
            'compare result in json format',
            resource_path=result_path,
            resource_type=resource_types.COMPILER_DEPENDENCIES_DIFF_RESULT,
        )
        self.mark_resource_ready(res.id)

        res = self.create_resource(
            'human readable missing deps sorted by popularity',
            resource_path=popularity_path,
            resource_type=resource_types.COMPILER_DEPENDENCIES_DIFF,
        )
        self.mark_resource_ready(res.id)
        popularity_link = channel.sandbox.get_resource_http_links(res.id)[0]

        res = self.create_resource(
            'human readable missing deps sorted by target',
            resource_path=targets_path,
            resource_type=resource_types.COMPILER_DEPENDENCIES_DIFF,
        )
        self.mark_resource_ready(res.id)
        targets_link = channel.sandbox.get_resource_http_links(res.id)[0]

        # now let's compare result with previous one
        if old_results:
            old_result_res = old_results[0]
            old_result_path = self.sync_resource(old_result_res)
            old_result = json.load(open(old_result_path))
            logging.debug("found old result in resource {res}, path={path}".format(
                res=old_result_res,
                path=old_result_path,
            ))
            result_delta_path = self.abs_path('result_delta.txt')
            result_delta = compare_dependencies.compare_results(
                old_result,
                result,
            )
            open(result_delta_path, 'w').write(result_delta)
            logging.debug("result_delta has size {len}".format(len=len(result_delta)))

            res = self.create_resource(
                'human readable delta between diff results',
                resource_path=result_delta_path,
                resource_type=resource_types.COMPILER_DEPENDENCIES_DIFF,
            )
            self.mark_resource_ready(res.id)
            result_delta_link = channel.sandbox.get_resource_http_links(res.id)[0]

            if result_delta:
                old_result_stats = compare_dependencies.calc_result_stats(old_result)
                result_stats = compare_dependencies.calc_result_stats(result)
                message = textwrap.dedent('''
                    There is delta in untracked dependencies since last check
                    Targets with missing dependencies: {targets} ({targets_delta:+})
                    Unique dependencies: {unique_deps} ({unique_deps_delta:+})
                    Total dependencies (sum by targets): {total_deps} ({total_deps_delta:+})
                    Diff: {result_delta_link}
                    Deps by targets: {targets_link}
                    Deps by popularity: {popularity_link}
                    Sandbox task: https://sandbox.yandex-team.ru/sandbox/tasks/view?task_id={task_id}
                ''').format(
                    targets=result_stats['targets'],
                    unique_deps=result_stats['unique_deps'],
                    total_deps=result_stats['total_deps'],
                    targets_delta=result_stats['targets']-old_result_stats['targets'],
                    unique_deps_delta=result_stats['unique_deps']-old_result_stats['unique_deps'],
                    total_deps_delta=result_stats['total_deps']-old_result_stats['total_deps'],
                    result_delta_link=result_delta_link,
                    targets_link=targets_link,
                    popularity_link=popularity_link,
                    task_id=self.id,
                ).strip()

                channel.sandbox.send_email(
                    ['arc-builds-dev'],
                    ['wawaka'],
                    'Compiler depends check report r{rev}'.format(rev=revision),
                    message,
                )
