# coding: utf-8

import os
import zlib
import time
import base64
import logging
import calendar
import aniso8601

from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia

from sandbox import sdk2


MAX_ALLOWED_COMMIT_DELAY = 12 * 60 * 60


class PatchParameter(SandboxStringParameter):
    name = 'patch'
    description = 'Apply patch'
    default_value = ''
    multiline = True


class PatchRootParameter(SandboxStringParameter):
    name = 'patch_root'
    description = 'Patch arcadia-root-relative root'
    default_value = ''
    multiline = False


class ArcadiaUrlParameter(SandboxStringParameter):
    name = 'arcadia_url'
    description = 'Merge patch with url (now must be trunk)'
    default_value = ''
    multiline = False


class AutocheckDiff(SandboxTask):
    type = 'AUTOCHECK_DIFF'
    execution_space = 20 * 1024  # 20 Gb

    input_parameters = [
        ArcadiaUrlParameter,
        PatchRootParameter,
        PatchParameter,
    ]

    def on_execute(self):
        arcadia_url = self.ctx.get(ArcadiaUrlParameter.name)

        parsed_arcadia_url = Arcadia.parse_url(arcadia_url)
        arcadia_info = Arcadia.info(arcadia_url)
        base_revision = int(arcadia_info['entry_revision'])

        utc_dt = aniso8601.parse_datetime(arcadia_info['date'])
        timestamp = calendar.timegm(utc_dt.timetuple())
        delta = time.time() - timestamp

        logging.info('Destination ts is %s, delta is %s', timestamp, delta)
        logging.info('Destination url is %s', parsed_arcadia_url)
        logging.info('Destination info is %s', arcadia_info)

        if parsed_arcadia_url.path != 'arc/trunk/arcadia':
            raise SandboxTaskFailureError('Destination url must be arc/trunk/arcadia')

        if delta > MAX_ALLOWED_COMMIT_DELAY:
            raise SandboxTaskFailureError('Destination commit too old, limit is {} hours'.format(MAX_ALLOWED_COMMIT_DELAY / 60 / 60))

        arcadia = Arcadia.get_arcadia_src_dir(arcadia_url)
        logging.info('Arcadia is fetched to %s', arcadia)

        patch = self.ctx.get(PatchParameter.name)
        if isinstance(patch, unicode):
            patch = patch.encode('utf-8', 'ignore')

        with open(self.log_path('patch.out'), 'w') as fp:
            fp.write(patch)

        patch_root = self.ctx.get(PatchRootParameter.name)
        Arcadia.apply_patch(os.path.join(arcadia, patch_root), patch, self.log_path())

        commit_time = int(time.time())
        commit_msg = self.ctx.get('commit_msg')
        review_id = self.ctx.get('review_id')
        author = self.author

        logging.info('Params are %s', dict(
            commit_time=commit_time,
            commit_msg=commit_msg,
            review_id=review_id,
            author=author,
        ))

        zipatch_path = self.log_path('zipatch.zip')
        ya_path = os.path.join(arcadia, 'ya')
        run_process([ya_path, '-v', 'check', '--add-all-files', '--create-zipatch', zipatch_path], work_dir=arcadia, wait=True, log_prefix='ya_check')

        with open(zipatch_path) as zf:
            zipatch = zf.read()

        review_autocheck_description = 'Diff check for Review <a href="https://a.yandex-team.ru/arc/review/{review_id}">{review_id}</a> by <a href="https://staff.yandex-team.ru/{author}">{author}@</a>'.format(review_id=review_id, author=author)
        review_autocheck_params = {
            'rev': base_revision,
            'zipatch': base64.b64encode(zipatch),
            'author': author,
            'patch': base64.b64encode(zlib.compress(patch)),
            'message': commit_msg,
            'gsid': self.ctx.get('gsid'),
        }

        if self.ctx.get('report'):
            review_autocheck_params.update({
                'report': review_id,
                'rb_creds': self.ctx.get('rb_token_vault_key'),
                'rb_creds_token': True,
            })

        review_autocheck_task_type = sdk2.Task['AUTOCHECK_FROM_REVIEW_BOARD_SLIM']
        review_autocheck_task_type(review_autocheck_task_type.current,
                                   owner=self.owner,
                                   description=review_autocheck_description,
                                   priority={'class': 'SERVICE', 'subclass': 'NORMAL'},
                                   notifications=[{'transport': 'email', 'statuses': ['FAILURE', 'EXCEPTION', 'TIMEOUT'], 'recipients': ['AUTOCHECK']}],
                                   **review_autocheck_params).enqueue()


__Task__ = AutocheckDiff
