import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.YMakeBase as ymb
import sandbox.projects.common.build.parameters as build_params
from sandbox.sandboxsdk.paths import make_folder
import sandbox.sandboxsdk.parameters as sb_params
from sandbox.sandboxsdk.process import run_process, check_process_timeout
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk import svn

from subprocess import call
import os.path
import json
import logging


class DebugMode(sb_params.SandboxBoolParameter):
    name = 'debug'
    description = 'Debug mode'
    default_value = False


class AutocheckDismissedOwners(ArcadiaTask):
    type = 'AUTOCHECK_DISMISSED_OWNERS'

    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = build_params.get_arcadia_params() + [
        DebugMode,
        build_params.EnvironmentVarsParam,
    ]

    def __init__(self, task_id=0):
        ArcadiaTask.__init__(self, task_id)

    def initCtx(self):
        self.execution_space = 10000  # 10Gb
        self.ctx['notify_via'] = 'email'

    def on_enqueue(self):
        ArcadiaTask.on_enqueue(self)
        self.set_info('task enqueued for execution')

    def write_json_file(self, fileName, info):
        path = fileName if os.path.isabs(fileName) else self.result_path(fileName)
        with open(path, "w") as res:
            json.dump(info, res, indent=2, sort_keys=True)

    def process_sandbox_tasks_dirs(self):
        sandbox_tasks_path = os.path.join(self.arcadia_src_dir, 'search', 'garden', 'sandbox-tasks', 'projects')

        all_sandbox_tasks_dirs = []
        sandbox_tasks_dirs_with_YMakeLists = []
        for root, dirs, files in os.walk(sandbox_tasks_path):
            root_relative = root[len(sandbox_tasks_path) + 1:]
            if not root_relative or root_relative == 'common':
                for d in dirs:
                    dir_for_check = os.path.join(root_relative, d)
                    all_sandbox_tasks_dirs.append(dir_for_check)
                    if os.path.isfile(os.path.join(sandbox_tasks_path, dir_for_check, 'ya.make')):
                        sandbox_tasks_dirs_with_YMakeLists.append(dir_for_check)

        script_with_owners = os.path.join(self.cur_dir, 'findAllYMakeListsWithOwners.sh')
        sandbox_YMakeLists_with_owners = self.result_path('_sandbox_tasks_YMakeLists_with_owners.txt')
        call([script_with_owners, sandbox_tasks_path, sandbox_YMakeLists_with_owners])

        with open(sandbox_YMakeLists_with_owners, 'r') as f:
            sandbox_tasks_dirs_with_owners = f.read().splitlines()

        self.write_json_file('_all_sandbox_tasks_dirs.json', all_sandbox_tasks_dirs)
        self.write_json_file('_sandbox_tasks_dirs_with_YMakeLists.json', sandbox_tasks_dirs_with_YMakeLists)
        self.write_json_file('_sandbox_tasks_dirs_with_owners.json', sandbox_tasks_dirs_with_owners)

        result = {}
        sandbox_dirs_without_owners = set(sandbox_tasks_dirs_with_YMakeLists) - set(sandbox_tasks_dirs_with_owners)
        result['without_owners'] = [os.path.join(d, 'ya.make') for d in sandbox_dirs_without_owners]
        sandbox_dirs_without_YMakeLists = set(all_sandbox_tasks_dirs) - set(sandbox_tasks_dirs_with_YMakeLists)
        result['without_cmakelists'] = list(sandbox_dirs_without_YMakeLists)

        self.write_json_file('sandbox_result.json', result)

    def process_dismissed_owners(self):
        scriptPath = os.path.join(self.cur_dir, 'findAllYMakeListsWithOwners.sh')
        foundYMakeLists = self.result_path('_all_YMakeLists_with_owners.txt')
        call([scriptPath, self.arcadia_src_dir, foundYMakeLists])

        with open(foundYMakeLists, 'r') as f:
            all_dirs = f.read().splitlines()

        all_logins_and_rbgroups = self.get_owners_logins_and_rbgroups()
        filtered_logins_and_rbgroups = {}
        for x in all_dirs:
            cur_dir = x.strip()
            if cur_dir in all_logins_and_rbgroups:
                filtered_logins_and_rbgroups[cur_dir] = all_logins_and_rbgroups[cur_dir]

        logins2lists = {}
        bad_logins2lists = {}
        groups2lists = {}
        for cur_dir, logins_and_rbgroups in filtered_logins_and_rbgroups.iteritems():
            all_owners = logins_and_rbgroups['logins'] + logins_and_rbgroups['rbgroups'] + \
                logins_and_rbgroups['bad_logins']
            for key, res_dict in [
                ('logins', logins2lists),
                ('bad_logins', bad_logins2lists),
                ('rbgroups', groups2lists)
            ]:
                for owner in logins_and_rbgroups[key]:
                    project = os.path.join(cur_dir, "ya.make")
                    if owner in res_dict:
                        res_dict[owner][project] = all_owners
                    else:
                        res_dict[owner] = {project: all_owners}

        groups_with_users = self.get_all_groups_with_users()

        non_existent_groups = {}
        for group in groups2lists.keys():
            if group not in groups_with_users:
                non_existent_groups[group] = groups2lists[group]

        logins2groups = {}
        small_groups = {}
        for group_name, group_info in groups_with_users.iteritems():
            logins = group_info.get('members', [])
            if len(logins) <= 1:
                small_groups[group_name] = logins

            for login in logins:
                if login in logins2groups:
                    logins2groups[login].append(group_name)
                else:
                    logins2groups[login] = [group_name]

        junk_path = self.abs_path('junk')
        svn.Arcadia.export('arcadia:/arc/trunk/arcadia/junk', junk_path, depth='immediates')
        logging.debug('junk content: %s', os.listdir(junk_path))
        junk_logins = [d for d in os.listdir(junk_path) if os.path.isdir(os.path.join(junk_path, d))]
        with open(os.path.join(self.cur_dir, 'SaveJunkList.txt'), "r") as fp:
            save_junk_list = fp.read().splitlines()

        for login in save_junk_list:
            if login in junk_logins:
                junk_logins.remove(login)

        all_logins = list(set(logins2lists.keys() + logins2groups.keys() + junk_logins))
        logging.debug('OWNERS request: ["{0}"]'.format('", "'.join(all_logins)))
        dismissed = self.get_dismissed_owners(all_logins)
        # maternity = self.get_maternity_owners(all_logins)
        maternity = {}
        for owner in all_logins:
            owner_not_dismissed = owner not in dismissed or not dismissed[owner]['is_dismissed']
            if owner_not_dismissed and owner not in maternity:
                if owner in logins2lists:
                    del logins2lists[owner]
                if owner in logins2groups:
                    del logins2groups[owner]
                if owner in junk_logins:
                    junk_logins.remove(owner)

                if owner in dismissed:
                    del dismissed[owner]

        self.write_json_file('dismissed_owners.json', dismissed)
        self.write_json_file('maternity.json', maternity)
        self.write_json_file('junk_logins.json', junk_logins)
        self.write_json_file('logins2lists.json', logins2lists)
        self.write_json_file('bad_logins2lists.json', bad_logins2lists)
        self.write_json_file('logins2groups.json', logins2groups)
        self.write_json_file('non_existent_groups.json', non_existent_groups)
        self.write_json_file('small_groups.json', small_groups)

        return {'dismissed': len(dismissed), 'maternity': len(maternity)}

    def result_path(self, p=""):
        return os.path.join(self.result_dir, p) if p else self.result_dir

    def run_cmd(self, cmd, cmdCtx):
        self.ctx[cmdCtx.logName + '_cmd'] = cmd
        logFile = self.result_path(cmdCtx.logName + cmdCtx.outputExt)
        errLogFile = self.result_path(cmdCtx.logName + cmdCtx.errOutputExt)

        with open(logFile, 'w') as output:
            with open(errLogFile, 'w') as errOutput:
                try:
                    cmdCtx.Verify()
                    p = run_process(
                        cmd, log_prefix=cmdCtx.logName, wait=cmdCtx.wait,
                        environment=cmdCtx.env, work_dir=cmdCtx.workDir,
                        stdin=cmdCtx.stdin, stdout=output, stderr=errOutput,
                        outputs_to_one_file=False
                    )
                except Exception as exc:
                    raise SandboxTaskFailureError(exc)
        return p, logFile

    def run_ymake(self, ymakeEx):
        return self.run_cmd(ymakeEx.ConstructCmd(), ymakeEx.cmdCtx)

    def get_owners_logins_and_rbgroups(self):
        ya_project_owners = ymb.YmakeWrapper(srcRoot=self.arcadia_src_dir, logName='ya_project_owners', subcommands=['project', 'owner'], use_autocheck_defines=False)
        ya_project_owners.args = ['--show-owners-and-groups', '--json-output', '-r', self.arcadia_src_dir]
        ya_project_owners.cmdCtx.workDir = self.arcadia_src_dir
        ya_project_owners.cmdCtx.outputExt = '.json'
        process, dump_file = self.run_ymake(ya_project_owners)
        check_process_timeout(process, 900, timeout_sleep=5)
        with open(dump_file) as dump:
            return json.load(dump)

    def get_all_groups_with_users(self):
        ya_dump_groups = ymb.YmakeWrapper(srcRoot=self.arcadia_src_dir, logName='ya_project_owners', subcommands=['dump', 'groups', 'groups_with_users'], use_autocheck_defines=False)
        ya_dump_groups.cmdCtx.workDir = self.arcadia_src_dir
        ya_dump_groups.cmdCtx.outputExt = '.json'
        process, dump_file = self.run_ymake(ya_dump_groups)
        check_process_timeout(process, 900, timeout_sleep=5)
        with open(dump_file) as dump:
            return json.load(dump)

    def get_dismissed_owners(self, all_logins):
        ya_dump_dismissed = ymb.YmakeWrapper(srcRoot=self.arcadia_src_dir, logName='ya_dump_dismissed', subcommands=['project', 'owner', 'check_logins'], use_autocheck_defines=False)
        ya_dump_dismissed.args = ['--is-dismissed'] + all_logins
        ya_dump_dismissed.cmdCtx.workDir = self.arcadia_src_dir
        ya_dump_dismissed.cmdCtx.outputExt = '.json'
        process, dump_file = self.run_ymake(ya_dump_dismissed)
        check_process_timeout(process, 900, timeout_sleep=5)
        try:
            with open(dump_file) as dump:
                return json.load(dump)
        except ValueError:
            logging.error('Unable to read json from file %s, file content: %s', dump_file, open(dump_file).read())
            raise

    def get_maternity_owners(self, all_logins):
        ya_dump_maternity = ymb.YmakeWrapper(srcRoot=self.arcadia_src_dir, logName='ya_dump_dismissed', subcommands=['project', 'owner', 'check_logins'], use_autocheck_defines=False)
        ya_dump_maternity.args = ['--is-maternity'] + all_logins
        ya_dump_maternity.cmdCtx.workDir = self.arcadia_src_dir
        ya_dump_maternity.cmdCtx.outputExt = '.json'
        process, dump_file = self.run_ymake(ya_dump_maternity)
        check_process_timeout(process, 900, timeout_sleep=5)
        with open(dump_file) as dump:
            return json.load(dump)

    def do_execute(self):
        self.result_dir = self.abs_path('result')
        make_folder(self.result_dir)

        self.arcadia_src_dir = self.get_arcadia_src_dir(ignore_externals=True)
        self.cur_dir = os.path.join(
            self.arcadia_src_dir,
            'sandbox', 'projects', 'AutocheckDismissedOwners'
        )

        self.process_sandbox_tasks_dirs()
        result_for_dashboard = self.process_dismissed_owners()
        self.write_json_file('result_for_dashboard.json', result_for_dashboard)

        resource = self._create_resource(
            "AutocheckDismissedOwners resource",
            self.result_dir,
            resource_types.AUTOCHECK_DISMISSED_OWNERS_RESULT,
        )
        resource.mark_ready()
        self.ctx['resource_id'] = resource.id
        self.ctx['resource_url'] = resource.http_url()


__Task__ = AutocheckDismissedOwners
