import os
import time
import json
import logging
import datetime as dt
import string
import random

import sandbox.common.types.client as ctc

import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.AutocheckStatistics import emulate
from sandbox.projects import resource_types
import sandbox.sandboxsdk.errors as sdk_errors
from sandbox import common
import sandbox.sandboxsdk.parameters as sdk_params
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment
import sandbox.sandboxsdk.svn as sdk_svn

import sandbox.projects.common.constants as consts
import sandbox.projects.common.time_utils as tu
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
import scenario_generator as sg

from sandbox import sdk2

DISTBUILD_TESTING_PROXIES = ['distbuild090.search.yandex.net:56000', 'distbuild091.search.yandex.net:56000']
ARCADIA_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'
DATE_FORMAT = '%Y-%m-%d %H:%M:%S'


class SandboxEmulator(object):
    def __init__(self, task):
        self.sandbox_rest_client = common.rest.Client()
        self.task = task
        self.n_tries = 12
        self.break_time = 0.1

    def retried(self, func, args):  # TODO: move this trash from here
        break_time = self.break_time
        for i in xrange(self.n_tries):
            try:
                return func(*args)
            except Exception as ex:
                logging.warning(ex)
                if i + 1 == self.n_tries:
                    raise ex
                time.sleep(break_time)
                break_time *= 1.5

    def get_task(self, task_id):
        response = self.sandbox_rest_client.task.read(limit=1, id=[task_id])
        if response['total'] != 1:
            logging.warning('Cannot find task with id %s', task_id)
            return {}

        task = response['items'][0]
        task_context = self.sandbox_rest_client.task[task_id].context.read()
        task_context.update(task['input_parameters'])
        task['ctx'] = task_context

        return task

    def add_autocheck_build_fallback(self, revision, description, shooting_params):
        input_parameters = {
            'make_context_on_distbuild_requirements_map': '{"default": {"mlock": 150, "cpu": 4}}',
            consts.ARCADIA_URL_KEY: 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia@{}'.format(revision),
            'autocheck_build_task': 'AUTOCHECK_BUILD_YA_2',
            'autocheck_revision': revision,
            'notify_if_failed': self.task.author,
            'notify_if_finished': '',
            'do_not_restart': True,
            'ymake_task_count': 0,
            'build_arch': 'linux',
        }

        input_parameters.update(shooting_params)

        autocheck_task_class = sdk2.Task['AUTOCHECK_BUILD_PARENT_2']
        autocheck_task = autocheck_task_class(
            autocheck_task_class.current,
            description=description,
            priority=['SERVICE', 'NORMAL'],
            inherit_notifications=False,
            **input_parameters
        )
        autocheck_task.enqueue()

        return autocheck_task.id

    def get_tasks(self, tasks_ids, fields):
        enable_context = False
        if 'ctx' in fields:
            enable_context = True
            fields.remove('ctx')
            fields.append('input_parameters')

        response = self.sandbox_rest_client.task.read(limit=len(tasks_ids), id=tasks_ids, fields=fields)
        if response['total'] != len(tasks_ids):
            found_ids = [t['id'] for t in response['items']]
            logging.warning('Cannot find tasks with ids: %s', set(tasks_ids) - set(found_ids))

        if enable_context:
            for item in response['items']:
                item['ctx'] = self.sandbox_rest_client.task[item['id']].context.read()
                item['ctx'].update(item['input_parameters'])

        return {t['id']: t for t in response['items']}

    def list_resources(self, params):
        response = self.sandbox_rest_client.resource.read(**params)
        return response['items']

    def get_resource_http_links(self, resource_id):
        return ['https://proxy.sandbox.yandex-team.ru/{}'.format(resource_id)]

    def update_context(self, **kwargs):
        time_delta = kwargs.get('time_delta', 0)
        self.task.ctx['continue_scenario_from'] += int(time_delta)


def retried_func(func):
    def retried_func(*args, **kwargs):
        count = 5
        is_last = lambda idx: idx == count - 1
        for i in xrange(count):
            try:
                logging.debug('Running {} with args: {}, kwargs: {}'.format(func.func_name, args, kwargs))
                result = func(*args, **kwargs)
                logging.debug('Success: {}'.format(str(result)))
                return result
            except Exception as ex:
                if is_last(i):
                    raise ex
                logging.debug(ex)
                time.sleep(i)
                continue
    return retried_func


class ScenarioShootingTask(SandboxTask):
    type = 'AUTOCHECK_EMULATION_TASK'
    cores = 1
    required_ram = 2048
    execution_space = 10 * 1024

    client_tags = ctc.Tag.LINUX_PRECISE

    environment = [
        PipEnvironment('matplotlib', '1.4', use_wheel=True),
        PipEnvironment('pandas', '0.16.2', use_wheel=True),
        PipEnvironment('python-dateutil'),
        PipEnvironment('inlinestyler'),
        PipEnvironment('Jinja2'),
        PipEnvironment('numpy'),
    ]

    class WayOfGettingScenario(sdk_params.SandboxStringParameter):
        name = 'way_of_getting_scenario'
        description = 'Get scenario from:'
        choices = [
            ('string', 'from_string'),
            ('torrent', 'from_torrent'),
            ('autocheck', 'generate_from_autocheck'),
            ('last revisions', 'generate_last_revisions_scenario'),
        ]
        sub_fields = {
            'from_string': ['scenario'],
            'from_torrent': ['torrent'],
            'generate_from_autocheck': ['scenario_scale_factor', 'start_delay', 'start_date', 'end_date'],
            'generate_last_revisions_scenario': ['last_revisions_count', 'last_revisions_period', 'last_revisions_step'],
        }
        default_value = 'from_string'

    class LastRevisionsCount(sdk_params.SandboxIntegerParameter):
        name = 'last_revisions_count'
        description = 'Revisions count'
        default_value = 10

    class LastRevisionsPeriod(sdk_params.SandboxIntegerParameter):
        name = 'last_revisions_period'
        description = 'Scheduling period'
        default_value = 60

    class LastRevisionsStep(sdk_params.SandboxIntegerParameter):
        name = 'last_revisions_step'
        description = 'Revision step'
        default_value = 1

    class ScenarioTorrent(sdk_params.SandboxStringParameter):
        name = 'torrent'
        description = 'Scenario rbtorrent'

    class Scenario(sdk_params.SandboxStringParameter):
        name = 'scenario'
        multiline = True
        description = 'Scenario'

    class ScenarioScaleFactor(sdk_params.SandboxFloatParameter):
        name = 'scenario_scale_factor'
        description = 'Scenario scale factor is used to adjust the duration of scenario'
        default_value = 1

    class StartScenarioDelay(sdk_params.SandboxIntegerParameter):
        name = 'start_delay'
        description = 'Delay (in seconds) before start scenario'
        default_value = 3600

    class StartDate(sdk_params.SandboxStringParameter):
        name = 'start_date'
        description = 'Start date ({})'.format(DATE_FORMAT)

    class EndDate(sdk_params.SandboxStringParameter):
        name = 'end_date'
        description = 'end date ({})'.format(DATE_FORMAT)

    class SendTo(sdk_params.SandboxStringParameter):
        name = 'send_to'
        description = 'Send report to (comma separated)'

    class DrawPictures(sdk_params.SandboxBoolParameter):
        name = 'draw_pictures'
        description = 'Draw pictures'
        default_value = True

    class DrawBoxes(sdk_params.SandboxBoolParameter):
        name = 'draw_boxes'
        description = 'Draw boxes'
        default_value = True

    class ShootingParams(sdk_params.SandboxStringParameter):
        name = 'shooting_params'
        description = 'Shooting tasks params'
        default_value = '{\n}'
        multiline = True

    input_parameters = [
        WayOfGettingScenario,
        Scenario,
        ScenarioTorrent,
        ScenarioScaleFactor,
        ShootingParams,
        build_params.JsonPrefix,
        StartScenarioDelay,
        StartDate,
        LastRevisionsCount,
        LastRevisionsPeriod,
        LastRevisionsStep,
        EndDate,
        DrawPictures,
        DrawBoxes,
        SendTo,
    ]

    def publish_images(self, images_dir):
        resource = self.create_resource(
            'images',
            resource_path=images_dir,
            resource_type=resource_types.AUTOCHECK_PLOT,
        )
        return 'https://proxy.sandbox.yandex-team.ru/{}'.format(resource.id)

    def generate_scenario_by_last_revisions(self, scenario_path):
        count = int(self.ctx[self.LastRevisionsCount.name])
        period = int(self.ctx[self.LastRevisionsPeriod.name])
        step = int(self.ctx[self.LastRevisionsStep.name])
        info = sdk_svn.Svn.info(ARCADIA_URL)
        revision = int(info['entry_revision'])
        with open(scenario_path, 'w') as scenario_file:
            scenario_file.write(sg.generate_scenario_by_last_revisions(revision, step, count, period))

    def generate_scenario_by_autocheck(self, scenario_path):
        start_date = self.ctx['start_date'].strip()
        end_date = self.ctx['end_date'].strip()
        if start_date and end_date:
            start_date = dt.datetime.strptime(start_date, DATE_FORMAT)
            end_date = dt.datetime.strptime(end_date, DATE_FORMAT)
        elif start_date or end_date:
            err_params = ('start date', start_date, 'end date') if start_date else ('end date', end_date, 'start date')
            raise sdk_errors.SandboxTaskFailureError('If {} specified ( = "{}") then {} must be specified too.'.format(*err_params))
        else:
            scenario_for_date = dt.date.today()
            if dt.datetime.now().hour < 21:
                scenario_for_date -= dt.timedelta(days=1)
            start_date = dt.datetime.combine(scenario_for_date, dt.datetime.min.time()) + dt.timedelta(hours=14)
            end_date = dt.datetime.combine(scenario_for_date, dt.datetime.min.time()) + dt.timedelta(hours=23)

        logging.info('Automatically generate scenario for %s - %s', start_date, end_date)
        log = sdk_svn.Svn.log(ARCADIA_URL, start_date.strftime('{%Y-%m-%dT%H:%M:%S}'), end_date.strftime('{%Y-%m-%dT%H:%M:%S}'))
        with open(scenario_path, 'w') as scenario_file:
            scenario_file.write(sg.generate_scenario_by_autocheck(log, scale_factor=self.ctx['scenario_scale_factor'], delay=self.ctx['start_delay']))

    def get_timeout_time(self, sandboxEmulator):
        if self.scheduler is not None:
            scheduler = sandboxEmulator.sandbox_rest_client.scheduler[abs(self.scheduler)].read()
            logging.info("Rest client answer: %s", scheduler)
            self.ctx['scheduler_start_time'] = self.ctx.get('scheduler_start_time') or scheduler['time']['last']
            start_time = dt.datetime.strptime(self.ctx['scheduler_start_time'], '%Y-%m-%dT%H:%M:%S.%fZ')
            timeout_at = start_time + dt.timedelta(seconds=self.ctx.get('kill_timeout'))
            logging.info("Start time(last scheduler run) = %s; end time = %s; now = %s", start_time, timeout_at, dt.datetime.utcnow())
            if timeout_at < dt.datetime.utcnow():
                raise sdk_errors.SandboxTaskFailureError('Activate scheduler timeout. Scheduler start time = {}, end time = {}'.format(start_time, timeout_at))
            return timeout_at

    def on_execute(self):
        sandboxEmulator = SandboxEmulator(self)
        timeout_at = self.get_timeout_time(sandboxEmulator)
        self.ctx['continue_scenario_from'] = self.ctx.get('continue_scenario_from', 0)

        description = 'Shooting {0} '.format(str(self.id))
        shooting_params = json.loads(self.ctx.get(self.ShootingParams.name))

        scenario_path = self.log_path('scenario.txt')
        if self.ctx.get(self.WayOfGettingScenario.name) == 'from_string' or self.ctx.get('continue_scenario_from'):
            scenario_str = self.ctx['scenario']
            if self.ctx.get('continue_scenario_from'):
                logging.info('Continue scenario from %s second', self.ctx.get('continue_scenario_from'))
                scenario_str = sg.continue_scenario(scenario_str, self.ctx.get('continue_scenario_from'))
            else:
                logging.info('Get scenario from string')
            with open(scenario_path, 'w') as scenario_file:
                scenario_file.write(scenario_str)
        elif self.ctx.get(self.WayOfGettingScenario.name) == 'from_torrent':
            logging.info('Get scenario from torrent')
            torrent = self.ctx['torrent']
            self.remote_copy(torrent, '.', 'skynet')
        elif self.ctx.get(self.WayOfGettingScenario.name) == 'generate_from_autocheck':
            self.generate_scenario_by_autocheck(scenario_path)
            with open(scenario_path, 'r') as scenario_file:
                self.ctx['scenario'] = scenario_file.read()

            random_str = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(4))
            self.ctx['json_prefix'] = 'shooting-{}-'.format(random_str)
        elif self.ctx.get(self.WayOfGettingScenario.name) == 'generate_last_revisions_scenario':
            self.generate_scenario_by_last_revisions(scenario_path)
            with open(scenario_path, 'r') as scenario_file:
                self.ctx['scenario'] = scenario_file.read()
        else:
            raise Exception('Unknown scenario')

        if self.ctx['json_prefix']:
            shooting_params['json_prefix'] = self.ctx['json_prefix']

        output_filename = 'OUT.json'
        time_to_wait = 0

        start_emulation_time = time.time()
        try:
            emulate.main(
                scenario_path,
                output_filename,
                None,
                None,
                None,
                time_to_wait,
                description,
                sandbox=sandboxEmulator,
                shooting_params=shooting_params,
                start_time=start_emulation_time,
                timeout_at=timeout_at,
                download_testenv_result=False,
            )
        except Exception:
            self.ctx['continue_scenario_from'] += int(time.time() - start_emulation_time)
            raise
        self.ctx['continue_scenario_from'] += int(time.time() - start_emulation_time)

        child_tasks = self.list_subtasks(load=False, completed_only=False, hidden=False)
        info = emulate.get_info(sandboxEmulator, child_tasks, download_testenv_result=False)

        with arcadia_sdk.mount_arc_path(
            "arcadia:/arc/trunk/arcadia/devtools/autocheck/stat", use_arc_instead_of_aapi=True
        ) as script_root:
            te_report_script = os.path.join(script_root, 'make_report.py')

            dump_path = self.abs_path('dump.csv')
            json_path = self.abs_path('OUT.json')
            html_path = self.log_path('report.html')
            te_html_path = self.log_path('te_report.html')
            emulate.write_info(info, json_path, html_path, dump_path, task=self, te_html_path=te_html_path, te_report_script=te_report_script, use_jinja_html_report=False)

            self.create_resource(
                'csv dump',
                resource_path=dump_path,
                resource_type=resource_types.AUTOCHECK_CSV_STAT,
            )

            self.create_resource(
                'tasks info',
                resource_path=json_path,
                resource_type=resource_types.AUTOCHECK_EMULATION_JSON,
            )

            send_to = self.ctx['send_to'].strip().split(',')
            if send_to:
                with open(te_html_path, 'r') as stream:
                    text = stream.read().strip()

                text += '\nTask: <a href="http://sandbox.yandex-team.ru/sandbox/tasks/view?task_id={}">{}</a>\n'.format(self.id, self.id)
                logging.info('Sending email to %s', send_to)
                channel.sandbox.send_email(
                    send_to,
                    '',
                    'Autocheck Emulation Report: {}'.format(tu.date_ymdhm()),
                    text,
                    content_type='text/html'
                )
                logging.info('Email sent')

    def on_before_timeout(self, seconds):
        subtasks = list(sdk2.Task.find(parent=sdk2.Task[self.id]).limit(1000))
        logging.info('Stop subtasks before timeout: %s', subtasks)
        for task in subtasks:
            task.stop()
        super(ScenarioShootingTask, self).on_before_timeout(seconds)

    def timeout_checkpoints(self):
        return [10]


__Task__ = ScenarioShootingTask
