import json
import logging
import os.path
import datetime as dt


from sandbox import sdk2
from sandbox.common import telegram
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import make_folder
import sandbox.sandboxsdk.parameters as sb_params

import responsibility


class ARC_RESPONSIBILITY_RESULT(sdk2.resource.AbstractResource):
    calc_md5 = False
    any_arch = True


class QAFW_RESPONSIBILITY_RESULT(sdk2.resource.AbstractResource):
    calc_md5 = False
    any_arch = True


class DEVTOOLS_TESTS_RESPONSIBILITY_RESULT(sdk2.resource.AbstractResource):
    calc_md5 = False
    any_arch = True


class DISTBUILD_RESPONSIBILITY_RESULT(sdk2.resource.AbstractResource):
    calc_md5 = False
    any_arch = True


class AUTOCHECK_SB_TASKS_RESPONSIBILITY_RESULT(sdk2.resource.AbstractResource):
    calc_md5 = False
    any_arch = True


class CONTRIB_RESPONSIBILITY_RESULT(sdk2.resource.AbstractResource):
    calc_md5 = False
    any_arch = True


class Responsibles(sb_params.SandboxStringParameter):
    name = 'responsibles'
    required = True
    description = 'Responsibles'
    default_value = 'snowball, kakabba, nalpp, rnefyodov'


class StartDate(sb_params.SandboxStringParameter):
    name = 'start_date'
    required = True
    description = 'Start date'
    default_value = '09.10.2014'


class StartFrom(sb_params.SandboxStringParameter):
    name = 'start_from'
    required = True
    description = 'Start from'
    default_value = 'snowball'


class PostToTelegram(sb_params.SandboxBoolParameter):
    name = 'post_to_telegram'
    required = False
    description = 'Post to telegram?'
    default_value = False


class ResponsibilityResultResourceType(sb_params.SandboxStringParameter):
    name = 'responsibility_result_resource_type'
    description = 'Responsibility result resource type'
    default_value = 'AUTOCHECK_RESPONSIBILITY_RESULT'


class TelegramChatId(sb_params.SandboxIntegerParameter):
    name = 'telegramchatid'
    description = 'Telegram chat id'
    default_value = -1001061146614  # DEVTOOLS


class AutocheckResponsibility(SandboxTask):
    type = 'AUTOCHECK_RESPONSIBILITY'

    RESOURCE_TYPE_TO_TITLE_MAPPING = {
        'AUTOCHECK_RESPONSIBILITY_RESULT': 'Duty reminder: *devtools@ mailing list and st/DEVTOOLSUP*',
        'DISTBUILD_RESPONSIBILITY_RESULT': 'Duty reminder: *distbuild*',
        'AUTOCHECK_SB_TASKS_RESPONSIBILITY_RESULT': 'Duty reminder: *autocheck*',
        'QAFW_RESPONSIBILITY_RESULT': 'Duty reminder: all our services (https://wiki.yandex-team.ru/qafw)',
        'DEVTOOLS_TESTS_RESPONSIBILITY_RESULT': 'Duty reminder: tests (https://wiki.yandex-team.ru/yatool/test/duty)',
        'CONTRIB_RESPONSIBILITY_RESULT': 'Duty reminder: *CONTRIB queue*',
        'ARC_RESPONSIBILITY_RESULT': 'Duty reminder: *arc*',
    }

    input_parameters = [
        ResponsibilityResultResourceType,
        TelegramChatId,
        Responsibles,
        StartDate,
        StartFrom,
        PostToTelegram,
    ]

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)

    def initCtx(self):
        self.ctx['notify_via'] = 'email'

    def write_json_file(self, fileName, info):
        with open(os.path.join(self.resultDirName, fileName), "w") as res:
            json.dump(info, res, indent=2, sort_keys=True)

    def post_to_telegram(self, title, result):
        try:
            bot_token = self.get_vault_data("YATOOL", "unitybot_telegram_token")
            bot = telegram.TelegramBot(bot_token)
            text = responsibility.week_responsibility(result)
            bot.send_message(self._get_ctx_value(TelegramChatId), '\n'.join([title, text]), parse_mode=telegram.ParseMode.MARKDOWN)
            logging.debug("AUTOCHECK_RESPONSIBILITY is finished")
        except Exception:
            logging.exception("Failed to send release notes to Telegram :-(", exc_info=True)

    def on_execute(self):
        logging.info("Start execute")

        if dt.datetime.today().weekday() in responsibility.WEEKEND:
            return

        start_from = self.ctx['start_from']
        duty_schedule = responsibility.generate_duty_schedule(self.ctx['responsibles'], start_from.strip())
        result = responsibility.generate_result(self.ctx['start_date'], duty_schedule)

        self.resultDirName = self.abs_path('result')
        make_folder(self.resultDirName)
        self.write_json_file('result.json', result)

        resource_type_str = self._get_ctx_value(ResponsibilityResultResourceType)
        resource = self._create_resource(
            "Responsibility resource",
            self.resultDirName,
            resource_type_str,
        )
        resource.mark_ready()
        self.ctx['resource_id'] = resource.id
        self.ctx['resource_url'] = resource.http_url()

        if self.ctx.get('post_to_telegram'):
            title = self.RESOURCE_TYPE_TO_TITLE_MAPPING.get(resource_type_str, "Duty reminder.")
            self.post_to_telegram(title, result)

    def _get_ctx_value(self, param):
        return self.ctx.get(param.name, param.default_value) or param.default_value


__Task__ = AutocheckResponsibility
