import re
import datetime as dt
import logging

WEEKEND = (5, 6)


def to_workday(day):
    # if start_d is weekend then start from Friday
    if day.weekday() in WEEKEND:
        day -= dt.timedelta(day.weekday() - 4)

    return day


def parse_responsibles(responsibles_str):
    resps = []
    duty_for_whole_week = True
    for r in re.split("[, ]*", responsibles_str.strip()):
        resp_data = r.split(":")
        if len(resp_data) == 1:
            resps.append(r)
            duty_for_whole_week = False
        else:
            resp = resp_data[0]
            duty_days = int(resp_data[1])
            duty_for_whole_week &= duty_days == 5
            resps += [resp for _ in range(duty_days)]
    return resps, duty_for_whole_week


def get_result_for_date(start_date, for_date, duty_schedule):
    def get_workdays_delta(start_day, end_day):
        end_work_day = to_workday(end_day)
        days_delta = (end_work_day - start_day).days
        if days_delta < 0:
            step = -1
            start_range = -1
            days_delta -= 1
        else:
            step = 1
            start_range = 1
            days_delta += 1

        logging.debug("{0} days delta; (step = {1}; start_range = {2})".format(days_delta, step, start_range))
        daygenerator = (start_day + dt.timedelta(x) for x in xrange(start_range, days_delta, step))
        return sum(step for day in daygenerator if day.weekday() not in WEEKEND)

    workdays_delta = get_workdays_delta(start_date, for_date)
    logging.debug(
        "%s workdays between %s and %s",
        workdays_delta,
        start_date.strftime('%d.%m.%Y'),
        for_date.strftime('%d.%m.%Y')
    )
    return for_date.strftime('%d.%m.%Y'), repr(for_date.weekday()), \
        duty_schedule[workdays_delta % len(duty_schedule)], for_date == dt.datetime.today().date()


def generate_result(start_date_str, duty_schedule):
    start = dt.datetime.strptime(start_date_str, '%d.%m.%Y')
    start_date = to_workday(start.date())

    res = []
    today = dt.datetime.today().date()
    week_start = today - dt.timedelta(today.weekday())

    start_from = week_start - dt.timedelta(3)
    for i in range(11):
        cur_date = start_from + dt.timedelta(i)
        res.append(get_result_for_date(start_date, cur_date, duty_schedule))

    return res


def generate_duty_schedule(responsibles_str, start_from):
    def shift(seq, n):
        return seq[n:] + seq[:n]

    resp, duty_for_whole_week = parse_responsibles(responsibles_str)

    if start_from not in resp:
        raise Exception

    start_num = resp.index(start_from)
    resp = shift(resp, start_num)

    duty_schedule = []
    if len(resp) % 5:
        for i in range(len(resp)):
            duty_schedule.extend(resp)
    else:
        if duty_for_whole_week:
            duty_schedule.extend(resp)
        else:
            for i in range(len(resp)):
                duty_schedule.extend(shift(resp, i))

    for m in range(len(resp)):
        logging.info(duty_schedule[m*len(resp):(m+1)*len(resp)])

    return duty_schedule


# XXX: copy-paste from devtools/unitybot
def get_responsibles(result, index):
    prev = result[index - 1][2] if index > 0 else 'unknown@'
    current = result[index][2]
    return prev, current


# XXX: copy-paste from devtools/unitybot
def week_responsibility(result, today=None):
    today = today or dt.datetime.now().strftime('%d.%m.%Y')
    week_res = []
    for i in range(len(result)):
        _, current = get_responsibles(result, i)
        result_line = '{}: {}@'.format(result[i][0], current)
        if result[i][0] == today:
            result_line = '*{}*'.format(result_line)
        week_res.append(result_line)
    return '\n'.join(week_res)
