# -*- coding: utf-8 -*-

import json
import logging
import datetime

from sandbox.common.types.client import Tag

from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment

import report_maker


class BuildStatisticsTask(SandboxTask):
    type = "AUTOCHECK_STATISTICS_TASK"
    client_tags = Tag.LINUX_PRECISE
    cores = 1
    execution_space = 2000

    environment = (PipEnvironment('matplotlib', '1.4', use_wheel=True), )

    class SendTo(SandboxStringParameter):
        name = 'send_to'
        description = 'Send report to'

    class DrawPictures(SandboxBoolParameter):
        name = 'draw_pictures'
        description = 'Draw pictures'
        default_value = False

    class DrawBoxes(SandboxBoolParameter):
        name = 'draw_boxes'
        description = 'Draw boxes'
        default_value = False

    class ForDashboard(SandboxBoolParameter):
        name = 'dash'
        description = 'For dashboard'
        default_value = False

    input_parameters = [
        SendTo,
        DrawPictures,
        DrawBoxes,
        ForDashboard
    ]

    def publish_images(self, images_dir):
        resource = self.create_resource(
            'images',
            resource_path=images_dir,
            resource_type=resource_types.AUTOCHECK_PLOT,
        )
        return 'https://proxy.sandbox.yandex-team.ru/{}'.format(resource.id)

    def on_execute(self):
        dump_path = self.abs_path('dump.csv')

        result, text = report_maker.make_report(time_limit=None, dump_to_file=dump_path, task=self)

        result_path = self.abs_path("statistics.out.json")

        with open(result_path, "wt") as f:
            json.dump(result, f, indent=4)

        self.create_resource(
            "autocheck statistics",
            resource_path=result_path,
            resource_type=resource_types.AUTOCHECK_STATISTICS,
            attributes={
                "dash": str(self.ctx.get("dash", False)),
            },
        )
        self.create_resource(
            "csv dump",
            resource_path=dump_path,
            resource_type=resource_types.AUTOCHECK_CSV_STAT,
        )

        send_to = self.ctx['send_to']

        if send_to:
            logging.debug('Send email to {}'.format(send_to))

            channel.sandbox.send_email(
                send_to,
                '',
                datetime.datetime.now().strftime("Autocheck Statistics Report: %Y-%m-%d %H:%M"),
                text,
                content_type='text/html',
                extra_headers=['reply-to: ymake-dev@yandex-team.ru']
            )


__Task__ = BuildStatisticsTask
