import time
import xmlrpclib
import urllib2
import datetime as dt
from optparse import OptionParser
from collections import defaultdict
import logging
import json
import getpass
from report_maker import output_stats, scenario_dict_to_data_frame, make_te_html_report
from json_trickery import transform_ctx_to_testenv_json
from data_frame import write_csv
import sandbox.projects.common.constants as consts


DONE_STATUSES = ('TIMEOUT', 'SUCCESS', 'STOPPED', 'RELEASED', 'EXCEPTION', 'DELETED', 'FAILURE', 'NO_RES')
SUCCESSFULLY_DONE_STATUSES = ('SUCCESS')
_TASKS_CHUNK_SIZE = 100


def get_scenario(filename):
    logging.info('Reading scenario from {0}...'.format(filename))
    with open(filename, 'r') as stream:
        data = map(int, stream.read().strip().split())
        scenario = zip(data, data[1:])[::2]
    logging.info('Reading scenario is done.')
    logging.info('Scenario (length {}) is:\n{}\n'.format(len(scenario), json.dumps(scenario, indent=4)))
    return scenario


class SandboxEmulator(object):
    class OAuthTransport(xmlrpclib.Transport):
        def __init__(self, token):
            xmlrpclib.Transport.__init__(self)
            self.token = token

        def send_request(self, connection, handler, request_body):
            xmlrpclib.Transport.send_request(self, connection, handler, request_body)
            if self.token:
                connection.putheader('Authorization', 'OAuth %s' % self.token)

    def __init__(self, path, token, username):
        self.proxy = xmlrpclib.ServerProxy(path, allow_none=True, transport=self.OAuthTransport(token))
        self.path = path
        self.username = username
        self.n_tries = 25
        self.break_time = 0.1

    def retried(self, func, args):
        break_time = self.break_time
        for i in xrange(self.n_tries):
            try:
                return func(*args)
            except Exception as ex:
                logging.warning(ex)
                if i + 1 == self.n_tries:
                    raise ex
                time.sleep(break_time)
                break_time *= 1.5

    def create_task(self, params):
        logging.info('Creating new task {0}'.format(params['type_name']))
        new_task_id = self.retried(self.proxy.create_task, args=(params, ))
        logging.info('Created {0} task.'.format(new_task_id))
        return new_task_id

    def get_task(self, task_id):
        return self.retried(self.proxy.get_task, args=(task_id, ))

    def get_children(self, task_id):
        return self.retried(self.proxy.list_tasks, args=({'parent_id': task_id, 'load': False}, ))

    def add_autocheck_build_fallback(
        self, revision, json_prefix, directory='util', description='TEST', use_testing_cluster=False,
        testing_cluster_id=None, no_tests_on_distbuild=False, use_ya_dev=False, rebuild=False
    ):
        params = {
            'type_name': 'AUTOCHECK_BUILD_PARENT',
            'priority': ['SERVICE', 'NORMAL'],
            'owner': self.username,
            'descr': description,
            'arch': 'linux',
        }
        params['ctx'] = {'autocheck_build_task': 'AUTOCHECK_BUILD_YA',
                         'autocheck_make_only_dirs': directory,
                         'autocheck_revision': revision, 'build_arch': 'linux',
                         consts.ARCADIA_URL_KEY: 'svn+ssh://arcadia.yandex.ru/arc/trunk@{0}'.format(revision),
                         'distbs_priority': ['SERVICE', 'LOW'], 'ymake_priority': ['BACKGROUND', 'NORMAL'],
                         'distbs_timeout': 10800, 'ymake_timeout': 18000, 'kill_timeout': 21600, 'do_not_restart': True,
                         'distbuild_priority': -2, 'json_prefix': json_prefix,
                         'autocheck_distbs_testing_cluster': use_testing_cluster,
                         'autocheck_distbs_testing_cluster_id': testing_cluster_id,
                         'no_tests_on_distbuild': no_tests_on_distbuild,
                         'skip_tests_data': True,
                         'autocheck_ymake_rebuild': rebuild,
                         'autocheck_ya_dev_version': use_ya_dev}
        return self.create_task(params)

    def get_tasks(self, tasks_ids, fields):
        return self.retried(self.proxy.bulk_task_fields, args=(tasks_ids, fields))

    def list_resources(self, params):
        return self.retried(self.proxy.list_resources, args=(params, ))

    def get_resource_http_links(self, resource_id):
        return self.retried(self.proxy.get_resource_http_links, args=(resource_id, ))

    def update_context(self, **kwargs):
        return


def check_tasks(sandbox, tasks, statuses, output_filename):
    # TODO: check tasks not in DONE_STATUSES only
    new_statuses = []
    info = []
    fields = ['status', 'ctx']

    def chunks(lst, size):
        while lst:
            yield lst[:size]
            lst = lst[size:]

    tasks_info_dict = {}
    for tasks_chunk in chunks(tasks, _TASKS_CHUNK_SIZE):
        tasks_info_dict.update(sandbox.get_tasks(tasks_chunk, fields))

    for task in tasks:
        task_info = tasks_info_dict[task]
        task_status = task_info['status']
        new_statuses.append(task_status)
        info.append(task_info)

    if new_statuses == statuses:
        return new_statuses
    else:
        for task, old, new in zip(tasks, statuses, new_statuses):
            if old != new:
                logging.info('Task {0} changed {1} -> {2}'.format(task, old, new))
        done_info = [inf for inf, status in zip(info, new_statuses) if status in SUCCESSFULLY_DONE_STATUSES]
        if len(done_info):
            logging.info('Dumping tmp stats to {0}'.format(output_filename))
            with open(output_filename, 'w') as stream:
                output_stats(done_info, stream)
            logging.info('Dumping {0} is done.'.format(output_filename))
        logging.info('Statuses count changes: %s', len(new_statuses))
        return new_statuses


def get_task_testenv_result(sandbox, task_id):
    resources = []
    for _ in range(5):
        resources = sandbox.list_resources({'type': ['TEST_ENVIRONMENT_JSON_V2'], 'state': ['READY'], 'task_id': [task_id]})
        if not resources:
            logging.warn('Unable to find TEST_ENVIRONMENT_JSON resources in task %s', task_id)
            time.sleep(10)
        else:
            break

    if not resources:
        logging.error('Unable to find TEST_ENVIRONMENT_JSON resources in task %s', task_id)
        return None

    resource_id = resources[0]['id']
    result = None
    logging.debug('Try to download TEST_ENVIRONMENT_JSON resource %s', resource_id)
    try:
        result = sandbox.retried(lambda id: json.load(urllib2.urlopen(sandbox.get_resource_http_links(id)[0])), args=(resource_id, ))
        logging.debug('TEST_ENVIRONMENT_JSON resource %s downloaded', resource_id)
    except Exception as ex:
        logging.error('Unable to load resource http links, resource id is %s. Looks like sandbox host is died: %s', resource_id, ex)

    return result


def get_statuses_counts(result, type):
    counts = defaultdict(int)
    for test in filter(lambda r: r['type'] == type, result['results']):
        counts[test['status']] += 1
    return counts


def get_info(sandbox, tasks, download_testenv_result=True):
    tasks_info = []
    testenv_results = []
    errors_count = 0
    if download_testenv_result:
        for task in tasks:
            testenv_result = get_task_testenv_result(sandbox, task)
            if testenv_result is not None:
                tasks_info.append(sandbox.get_task(task))
                testenv_results.append(testenv_result)
            else:
                errors_count += 1

        for task_info, result in zip(tasks_info, testenv_results):
            task_info['testenv_statuses'] = {'targets': get_statuses_counts(result, 'build'), 'tests': get_statuses_counts(result, 'test')}
    else:
        tasks_info = [sandbox.get_task(task) for task in tasks]

    return tasks_info


def write_info(info, json_filename, html_filename, csv_filename, task=None, te_html_path=None, te_report_script=None, use_jinja_html_report=True):
    if json_filename:
        logging.info('Dumping resulting json to {0}'.format(json_filename))
        te_json = transform_ctx_to_testenv_json(info)
        with open(json_filename, 'w') as stream:
            json.dump(te_json, stream, indent=4, sort_keys=True)
        logging.info('Dumping {0} is done.'.format(json_filename))

    if json_filename and te_report_script:
        logging.info('Dumping TE html report to {0}'.format(te_html_path))
        make_te_html_report(json_filename, te_html_path, te_report_script)
        logging.info('Dumping {0} is done.'.format(te_html_path))

    if html_filename:
        logging.info('Dumping html report to {0}'.format(html_filename))
        with open(html_filename, 'w') as stream:
            output_stats(info, stream, use_jinja=use_jinja_html_report, task=task)
        logging.info('Dumping {0} is done.'.format(html_filename))

    if csv_filename:
        logging.info('Dumping csv report to {0}'.format(csv_filename))
        dump = scenario_dict_to_data_frame(info)
        with open('dump.csv', 'w') as stream:
            write_csv(dump, stream)
        logging.info('Dumping {0} is done.'.format(csv_filename))


def parse_args():
    parser = OptionParser()
    parser.add_option('-s', '--scenario', dest='scenario_filename', default='scenario.txt',
                      help='Scenario file.')
    parser.add_option('-o', '--output', dest='output_filename', default='out.txt',
                      help='Output file.')
    parser.add_option('-d', '--dir', dest='directory', default='util',
                      help='Directory to build("util" by default).')
    parser.add_option('-p', '--path', dest='path',
                      default='https://sandbox.yandex-team.ru/sandbox/xmlrpc',
                      help='Sandbox server path("https://sandbox.yandex-team.ru/sandbox/xmlrpc" by default).')
    parser.add_option('-w', '--wait', dest='time_to_wait', default=0,
                      help='Time to wait before start(0 by default).')
    parser.add_option('-t', '--token', dest='token', default='.token',
                      help='Token filename(".token" by default).')
    return parser.parse_args()


def main(scenario_filename, output_filename, path, username, token, time_to_wait, description, shooting_params, sandbox=None, start_time=None, timeout_at=None, download_testenv_result=True):
    time.sleep(time_to_wait)
    sandbox = sandbox or SandboxEmulator(path, token, username)
    scenario = get_scenario(scenario_filename)
    NOT_READY_STATUS = 'NOT_READY'
    current_tasks = []
    current_statuses = []
    is_started = [False for _ in scenario]
    start_time = start_time or time.time()

    while True:
        time.sleep(10)
        if timeout_at is not None and timeout_at < dt.datetime.utcnow():
            break
        time_delta = time.time() - start_time

        sandbox.update_context(time_delta=time_delta)
        # Checking whether new tasks have to be created
        for i in range(len(scenario)):
            (birth_time, revision), started = scenario[i], is_started[i]
            if time_delta > birth_time and not started:
                current_tasks.append(sandbox.add_autocheck_build_fallback(revision, description, shooting_params))
                current_statuses.append(NOT_READY_STATUS)
                is_started[i] = True
                logging.info('Task started, revision = %s, birth time = %s', revision, birth_time)

        # Checking statuses
        current_statuses = check_tasks(sandbox, current_tasks, current_statuses, 'tmp_stats.txt')

        # Checking whether all tasks are completed
        if len(current_statuses) == len(scenario) and not any(map(lambda status: status not in DONE_STATUSES, current_statuses)):
            break

    return get_info(sandbox, current_tasks, download_testenv_result=download_testenv_result)


if __name__ == '__main__':
    options, args = parse_args()
    scenario_filename = options.scenario_filename
    output_filename = options.output_filename
    path = options.path
    directory = options.directory
    time_to_wait = options.time_to_wait
    username = getpass.getuser()
    json_prefix = 'SCRIPT'
    logging.basicConfig(filename='logs.log', level=logging.DEBUG, format='%(levelname)s:%(message)s')

    with open(options.token, 'r') as stream:
        token = stream.read().strip()

    info = main(scenario_filename, output_filename, path, directory, username, token, time_to_wait, json_prefix)
    write_info(info, json_filename=output_filename, html_filename='report.html', csv_filename='dump.csv')
