import json
from optparse import OptionParser


stages = [
    "before_create_task",
    "task_created",
    "task_started",
    "svn_source_started",
    "svn_source_finished",
    "svn_testdata_started",
    "svn_testdata_finished",
    "configure_started",
    "configure_finished",
    "gen_graph_started",
    "gen_graph_finished",
    "build_started",
    "distbs_started",
    "distbs_scheduled",
    "distbs_queued",
    "distbs_running",
    "distbs_critical_copy_build",
    "distbs_critical_build_test",
    "distbs_ok",
    "distbs_finished",
    "relabel_finished",
    "relabel_started",
    "build_finished",
    "testing_started",
    "testing_finished",
    "build_task_finished",
    "task_finished",
    "task_processed",
    "notifications_sent",
]


def scenario_to_testenv_revision(revision, common_timestamp):
    testenv_revision = {}

    # empty cmake
    testenv_revision['cmake'] = {'email_log': {}}

    # timestamp
    if common_timestamp is not None:
        testenv_revision['timestamp'] = common_timestamp
    else:
        testenv_revision['timestamp'] = revision.get('ctx', {}).get('time_metrics', {}).get('commit_time', None)

    # distbuild
    testenv_revision['distbuild_linux'] = {}
    ctx = revision['ctx']
    if 'time_metrics' not in ctx:
        return testenv_revision
    else:
        time_metrics = ctx['time_metrics']
        shift = time_metrics.get('task_created', 0)
        for stage in stages:
            if time_metrics.get(stage):
                time_metrics[stage] -= shift

    distbuild = testenv_revision['distbuild_linux']
    distbuild.update(time_metrics)
    if 'build_system_used' in ctx:
        distbuild['build_system_used'] = ctx['build_system_used']

    return testenv_revision


def scenario_to_testenv_dict(data, common_timestamp=None):
    testenv_data = {}

    for task in data:
        task_ctx = task['ctx']
        revision = task_ctx['autocheck_revision']
        testenv_data[revision] = scenario_to_testenv_revision(task, common_timestamp)

    return testenv_data


def transform_ctx_to_testenv_json(data):
    def _transform(item):
        if item['status'] != 'SUCCESS':
            return None

        ts = min(*item['ctx']['__STAGES'].values())
        result = {
            "check_id": None,
            "data": {"email_log": {}},
            "revision": item['ctx']['autocheck_revision'],
            "stages": dict((k, v - ts) for k, v in item['ctx']['__STAGES'].items()),
            "task_id": item['id'],
            "timestamp": ts
        }

        result['stages']['notifications_sent'] = max(*result['stages'].values())
        return result

    return filter(None, [_transform(item) for item in data])


def transform_json(input_filename, output_filename):
    with open(input_filename, 'r') as stream:
        scenario_dict = json.load(stream)

    testenv_dict = scenario_to_testenv_dict(scenario_dict)

    with open(output_filename, 'w') as stream:
        json.dump(testenv_dict, stream, indent=4, sort_keys=True)


def parse_args():
    parser = OptionParser()
    parser.add_option('-i', '--input', dest='input_filename', help="Scenario's output json.")
    parser.add_option('-o', '--output', dest='output_filename', help="Testenv json emulation.")
    return parser.parse_args()


if __name__ == '__main__':
    options, args = parse_args()
    input_filename = options.input_filename
    output_filename = options.output_filename
    transform_json(input_filename, output_filename)
