import os
import sys
import json
import datetime
import logging
import subprocess
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class DrawCacheHitOptions(object):
    def __init__(self, **kwargs):
        self.limit = kwargs.get('limit')
        self.output_filename = kwargs.get('output_filename')
        self.dash = kwargs.get('dash', False)
        self.path = kwargs.get('path')

    def gen_command(self):
        command = []
        if self.limit:
            command += ['--limit={}'.format(self.limit)]
        if self.output_filename:
            command += ['--out={}'.format(self.output_filename)]
        if self.path:
            command += ['--path={}'.format(self.path)]
        if self.dash:
            command += ['--dashboard_sized']
        return command


class BoxplotterOptions(object):
    def __init__(self, **kwargs):
        self.limit = kwargs.get('limit', None)
        self.output = kwargs.get('output', None)
        self.input = kwargs.get('input', None)
        self.title = kwargs.get('title', None)

    def gen_command(self):
        command = []
        if self.limit:
            command += ['--limit={}'.format(self.limit), ]
        if self.output:
            command += ['--out={}'.format(self.output), ]
        if self.input:
            command += ['--in={}'.format(self.input), ]
        if self.title:
            command += ['--title={}'.format(self.title), ]
        return command


class DrawOptions(object):
    def __init__(self, **kwargs):
        self.task_id = kwargs.get('task_id', None)
        self.limit = kwargs.get('limit', None)
        self.date_from = kwargs.get('date_from', None)
        self.date_to = kwargs.get('date_to', None)
        self.output_filename = kwargs.get('output_filename', None)
        self.x_axis = kwargs.get('x_axis', 'timestamp')
        self.boxes = kwargs.get('boxes', False)
        self.use_all_periods = kwargs.get('use_all_periods', True)
        self.yheight = kwargs.get('yheight', None)
        self.depth = kwargs.get('depth', 2)
        self.wood = kwargs.get('wood', True)
        self.path = kwargs.get('path', None)
        self.buildtype = kwargs.get('buildtype', 'distbuild_linux')
        self.dash = kwargs.get('dash', False)
        self.disable_legend = kwargs.get('disable_legend', False)
        self.review = kwargs.get('review', False)
        self.rb_data = kwargs.get('rb_data', None)

    def gen_command(self):
        command = []
        if self.task_id:
            command += ['--emulation={}'.format(self.task_id), ]
        if self.limit:
            command += ['--limit={}'.format(self.limit), ]
        if self.date_from:
            command += ['--from="{}"'.format(self.date_from), ]
        if self.date_to:
            command += ['--to="{}"'.format(self.date_to), ]
        if self.output_filename:
            command += ['--out={}'.format(self.output_filename), ]
        if self.x_axis:
            command += ['--axis={}'.format(self.x_axis), ]
        if self.boxes:
            command += ['--boxes', ]
        if self.use_all_periods:
            command += ['--all', ]
        if self.wood:
            command += ['--wood', ]
        if self.yheight:
            command += ['--yheight={}'.format(self.yheight), ]
        if self.depth:
            command += ['--depth={}'.format(self.depth), ]
        if self.path:
            command += ['--path={}'.format(self.path), ]
        if self.buildtype:
            command += ['--buildtype={}'.format(self.buildtype), ]
        if self.dash:
            command += ['--dashboard_sized', ]
        if self.disable_legend:
            command += ['--disable_legend', ]
        if self.review:
            command += ['--review', ]
        if self.rb_data:
            command += ['--rb_data={}'.format(self.rb_data)]
        return command


def get_figures_dir():
    figures_dir = os.path.join(os.path.realpath(os.curdir), 'images')
    if not os.path.exists(figures_dir):
        os.mkdir(figures_dir)
    return figures_dir


def get_draw_path():
    plotter_svn_path = 'arcadia:/arc/trunk/arcadia/devtools/autocheck/analyze/plotter'
    plotter_local_path = os.path.join(os.path.realpath(os.curdir), 'plotter_dir')
    if not os.path.exists(plotter_local_path):
        Arcadia.export(plotter_svn_path, plotter_local_path)
    draw_path = os.path.join(plotter_local_path, 'draw.py')
    return draw_path


def get_cache_hit_draw_path():
    return os.path.join(os.path.split(get_draw_path())[0], 'draw_cache_hit.py')


def get_boxplotter_path():
    boxplotter_svn_path = 'arcadia:/arc/trunk/arcadia/devtools/autocheck/analyze/tests_plotter'
    boxplotter_local_path = os.path.join(os.path.realpath(os.curdir), 'boxplotter_dir')
    if not os.path.exists(boxplotter_local_path):
        Arcadia.export(boxplotter_svn_path, boxplotter_local_path)
    boxplotter_path = os.path.join(boxplotter_local_path, 'boxplotter.py')
    return boxplotter_path


def get_python_path():
    return sys.executable


def prepare_env(client_info):
    if '12.04-precise' in client_info['platform']:
        libstd_path = '/usr/lib/x86_64-linux-gnu/libstdc++.so.6'
        if os.path.exists(libstd_path):
            os.environ.update({'LD_PRELOAD': '/usr/lib/x86_64-linux-gnu/libstdc++.so.6', })
        else:
            raise SandboxTaskFailureError('{} not found'.format(libstd_path))


def draw_cache_hit_scatter(client_info, **kwargs):
    figure_path = os.path.join(get_figures_dir(), 'cache_hit_' + str(kwargs.get('limit')) + '.svg')
    kwargs.update({'output_filename': figure_path})

    prepare_env(client_info)
    command = [get_python_path(), get_cache_hit_draw_path(), ] + DrawCacheHitOptions(**kwargs).gen_command()

    full_command = ' '.join(command)

    logging.info('RUN {}'.format(full_command))
    process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    process.wait()
    out, err = process.communicate()
    logging.info('DONE: {0}\nOUT: {1}\nERR: {2}'.format(full_command, out, err))

    return figure_path


def draw_precommit_checks(client_info, **kwargs):
    figure_path = os.path.join(get_figures_dir(), 'rbcheck_' + str(kwargs.get('limit')) + ('_boxes' if kwargs.get('boxes') else '') + '.png')
    kwargs.update({'output_filename': figure_path})

    prepare_env(client_info)
    command = [get_python_path(), get_draw_path(), ] + DrawOptions(**kwargs).gen_command()

    full_command = ' '.join(command)

    logging.info('RUN {}'.format(full_command))
    process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    process.wait()
    out, err = process.communicate()
    logging.info('DONE: {0}\nOUT: {1}\nERR: {2}'.format(full_command, out, err))

    return figure_path


def draw_critical_path_boxplots(client_info, scores, filename):
    head = 15
    title = '{} most frequent critical path guys'.format(head)

    all_labels = sorted(scores.keys(), key=lambda k: len(scores[k]))[-head:]
    all_times = [scores[k] for k in all_labels]
    shortify = lambda txt, size: txt if len(txt) < size else txt[:size] + '...'
    rm_br = lambda txt: txt.replace('$(BUILD_ROOT)/', '')
    all_labels = map(lambda (label, times): shortify(rm_br(label), 120) + ' | ' + str(len(times)), zip(all_labels, all_times))

    figure_path = os.path.join(get_figures_dir(), filename)
    data_path = os.path.join(os.path.realpath(os.curdir), 'data.json')
    json.dump([all_labels, all_times], open(data_path, 'w'))

    prepare_env(client_info)

    command = [get_python_path(), get_boxplotter_path(), ] + BoxplotterOptions(input=data_path, output=figure_path, title=title).gen_command()
    full_command = ' '.join(command)

    logging.info('RUN {}'.format(full_command))
    process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    returncode = process.wait()
    out, err = process.communicate()
    logging.info('DONE: {0}\nOUT: {1}\nERR: {2}'.format(full_command, out, err))
    if returncode != 0:
        raise SandboxTaskFailureError('process {0} terminated with nonzero code {1}'.format(full_command, returncode))

    return figure_path


def make_plot(client_info, **kwargs):
    buildtype = kwargs.get('buildtype', 'distbuild_linux')
    now = datetime.datetime.now()
    date_from = kwargs.get('date_from', now)
    date_to = kwargs.get('date_to', now)
    figures_dir = get_figures_dir()
    to_str = lambda dt: dt.strftime('%b_%d_%H_%M')
    figure_name = '{0}_{1}_{2}.png'.format(buildtype, to_str(date_from), to_str(date_to))
    if kwargs.get('boxes'):
        figure_name = 'fancy_' + figure_name
    figure_path = os.path.join(figures_dir, figure_name)
    depth = 2 if 'dist' in buildtype else 1

    strtime = lambda dt: datetime.datetime.strftime(dt, '%Y-%m-%d %H:%M:%S')
    if 'date_from' in kwargs:
        kwargs['date_from'] = strtime(kwargs['date_from'])
    if 'date_to' in kwargs:
        kwargs['date_to'] = strtime(kwargs['date_to'])
    options = DrawOptions(output_filename=figure_path, depth=depth, **kwargs)

    python_path = get_python_path()
    draw_path = get_draw_path()
    command = [python_path, draw_path, ] + options.gen_command()
    full_command = ' '.join(command)
    prepare_env(client_info)
    logging.info('RUN: {}'.format(full_command))
    process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    returncode = process.wait()
    out, err = process.communicate()
    logging.info('DONE: {0}\nOUT: {1}\nERR: {2}'.format(full_command, out, err))
    if returncode != 0:
        raise SandboxTaskFailureError('process {0} terminated with nonzero code {1}'.format(full_command, returncode))

    return figure_path


def draw_fancies(data_frame, dump_path, client_info):
    buildtypes = set(data_frame['type'])
    period_length = datetime.timedelta(hours=3)
    date_to = datetime.datetime.now()
    date_from = date_to - period_length
    for buildtype in buildtypes:
        make_plot(dump_path, buildtype, date_from, date_to, client_info, boxes=True)
