import logging
import os
import sys
import subprocess
import datetime
import jinja2
import stat_maker
import time
import json
import testenv

from datetime import timedelta
from collections import defaultdict
from json_trickery import scenario_to_testenv_dict
from stat_maker import raw_data_to_data_frame
from plot_maker import get_figures_dir


def get_html_testenv_line(revision, task_url, testenv_statuses):
    statuses_map = {
        'OK': ('green', 'success'),
        'FAILED': ('red', 'failed'),
        'FLAKY': ('orange', 'flaky'),
        'SKIPPED': ('grey', 'skipped')
    }

    order = ('OK', 'FAILED', 'FLAKY', 'SKIPPED')

    def colorize(text, color):
        return '<font color={}>{}</font>'.format(color, text)

    def hlink(text, url):
        return '<a href="{}">{}</a>'.format(url, text)

    def group_results(group):
        statuses_counts = testenv_statuses[group]
        text = []
        for status in order:
            if status in statuses_counts:
                color, name = statuses_map[status]
                text.append(colorize('{} {}'.format(name, statuses_counts[status]), color))
        return '{}: '.format(group) + ', '.join(text)

    return '<p>' + ' | '.join([hlink('r{}'.format(revision), task_url), group_results('targets'), group_results('tests')]) + '</p>'


def append_html_testenv_lines(text, scenario_dict):
    for task in scenario_dict:
        text += '\n' + get_html_testenv_line(task['ctx']['autocheck_revision'], 'http://sandbox.yandex-team.ru/sandbox/tasks/view?task_id={}'.format(task['id']), task['testenv_statuses'])
    return text


def scenario_dict_to_data_frame(scenario_dict, common_timestamp=None):
    testenv_dict = scenario_to_testenv_dict(scenario_dict, common_timestamp=common_timestamp)
    data = raw_data_to_data_frame(testenv_dict)
    return data


def make_te_html_report(te_json, report_path, te_report_script):
    cmd = [sys.executable, te_report_script, te_json, report_path]
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    out, err = process.communicate()
    logging.debug('DONE: %s (exit code %s)\nOUT: %s\nERR: %s,', cmd, process.returncode, out, err)


def output_stats(scenario_dict, output_stream, use_jinja=False, task=None):
    """Transforms "emulate.py" json into stats table and outputs it."""
    now_timestamp = time.time()
    now = datetime.datetime.fromtimestamp(now_timestamp)
    delta = datetime.timedelta(seconds=10)
    buildtype = 'distbuild_linux'
    data = scenario_dict_to_data_frame(scenario_dict, common_timestamp=now_timestamp)
    if not use_jinja:
        make_table_report(data, buildtype, now - delta, now + delta, output_stream)
    else:
        result, text = make_report(dump_to_file='tmp_dump.csv', task=task, raw_data=scenario_to_testenv_dict(scenario_dict))
        output_stream.write(append_html_testenv_lines(text, scenario_dict))


def place_old_first(keys, old_keys):
    keys_map = defaultdict(list)
    out_of_old_keys = []
    for key in keys:
        if key in old_keys:
            keys_map[key].append(key)
        else:
            for old_key in old_keys:
                if key.startswith(old_key):
                    keys_map[old_key].append(key)
                    break
            else:
                out_of_old_keys.append(key)
    in_old_keys = []
    for old_key in old_keys:
        for key in keys_map[old_key]:
            in_old_keys.append(key)
    return in_old_keys + sorted(out_of_old_keys)


def standartize_keys(keys, percentile_keys):
    old_keys = [
        'checks',
        'revisions',
        'min_revision',
        'max_revision',
        'out-of-kpi',
        'fallbacks',
        'emails-count',
        'emails-count-owners',
        'emails-count-commit-authors',
        'emailed-revisions',
        'emailed-revisions-owners',
        'emailed-revisions-commit-authors',
        'emailed-revisions-out-of-kpi',
    ]

    old_percentile_keys = [
        'testenv-lag',
        'sandbox-testenv-lag',
        'sandbox-queue-lag',
        'svn-testdata-delay',
        'svn-source-delay',
        'configuring-time',
        'gen-graph-time',
        'distbs_worktime',
        'critical-path',
        'critical-path-copying',
        'critical-path-testing',
        'critical-path-building',
        'critical-path-svn-source',
        'build-time',
        'testing-time',
        'build-task-worktime',
        'wait-child-lag',
        'previous-await-lag',
        'notification-delay',
        'user-notification-delay',
        'precommit-check-delay',
        'sandbox-testenv-kpi',
    ]

    keys = place_old_first(keys, old_keys)
    percentile_keys = place_old_first(percentile_keys, old_percentile_keys)

    return keys, percentile_keys


def distribute_keys(stats):
    keys = []
    percentile_keys = []
    percentile_sign = 'p50'
    for key, value in stats.items():
        if value:
            if percentile_sign in value:
                if value[percentile_sign]:
                    percentile_keys.append(key)
            else:
                keys.append(key)
    return keys, percentile_keys


def get_keys_scheme(stats):
    keys_scheme = {}

    old_buildtypes = [
        'distbuild_linux',
        'cmake',
    ]

    old_periods = [
        'day',
        'week',
        'month',
    ]

    for buildtype, buildtime_stats in stats.items():
        keys_scheme[buildtype] = {}
        for period, period_stats in buildtime_stats.items():
            keys_scheme[buildtype][period] = {}
            keys, percentile_keys = distribute_keys(period_stats)
            keys, percentile_keys = standartize_keys(keys, percentile_keys)
            keys_scheme[buildtype][period]['keys'] = keys
            keys_scheme[buildtype][period]['percentile_keys'] = percentile_keys
        keys_scheme[buildtype]['ordered_periods'] = place_old_first(keys_scheme[buildtype].keys(), old_periods)
    keys_scheme['ordered_buildtypes'] = place_old_first(keys_scheme.keys(), old_buildtypes)
    return keys_scheme


def make_table_report(data, buildtype, from_date, to_date, stream):
    stats = stat_maker.calc_stat_for_buildtype_and_period(data, buildtype, from_date, to_date)
    keys, percentile_keys = distribute_keys(stats)
    percentiles = ['p50', 'p90', 'p95', 'p99']

    # single variables
    for key in keys:
        if 'current' in stats[key]:
            stream.write(key.ljust(40) + str(stats[key]['current']).rjust(30) + '\n')

    # percentiled variables head
    stream.write('variable'.ljust(40))
    for p in percentiles:
        stream.write(p.rjust(20))
    stream.write('\n')

    # percentiled variables values
    for key in percentile_keys:
        stream.write(key.ljust(40))
        for p in percentiles:
            if 'current' in stats[key][p]:
                stream.write(str(timedelta(seconds=int(stats[key][p]['current']))).rjust(20))
            else:
                stream.write(''.rjust(30))
        stream.write('\n')


def get_template(template_filename):
    template_path = os.path.dirname(os.path.abspath(__file__))
    env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
    return env.get_template(template_filename)


def publish_images(task):
    images_dir = get_figures_dir()
    return task.publish_images(images_dir)


def transform_resource_paths_into_urls(images, task):
    images_url = publish_images(task)
    new_images = {}
    for buildtype, periods in images.items():
        new_images[buildtype] = {}
        for period, images in periods.items():
            new_images[buildtype][period] = []
            for image_path in images:
                image_name = os.path.split(image_path)[-1]
                new_images[buildtype][period].append('/'.join([images_url, image_name]))
    logging.debug('Images urls are %s', json.dumps(new_images, indent=4, sort_keys=True))
    return new_images


def make_report(time_limit=None, dump_to_file=None, task=None, raw_data=None):
    draw_plot = task.ctx.get('draw_pictures', False) if task is not None else False
    draw_box = task.ctx.get('draw_boxes', False) if task is not None else False
    raw_data = raw_data or testenv.fetch(time_limit)
    if task is None or task.ctx.get('torrent') is None:
        result = stat_maker.run(time_limit, dump_to_file, task, raw_data)
    else:
        delta = datetime.timedelta(hours=1)
        date_from = datetime.datetime.fromtimestamp(int(min([el['timestamp'] for el in raw_data.values()]))) - delta
        date_to = datetime.datetime.fromtimestamp(int(max([el['timestamp'] for el in raw_data.values()]))) + delta
        result = stat_maker.run(task=task, raw_data=raw_data, buildtypes=['distbuild_linux', ], periods={'emulation': (date_from, date_to)})
    keys_scheme = get_keys_scheme(result['stat'])
    result['images']['cache_hit'] = transform_resource_paths_into_urls(result['images']['cache_hit'], task)
    if draw_plot:
        result['images']['plot'] = transform_resource_paths_into_urls(result['images']['plot'], task)
    if draw_box:
        result['images']['box'] = transform_resource_paths_into_urls(result['images']['box'], task)
    if 'rb_checks' in result['images']:
        result['images']['rb_checks'] = transform_resource_paths_into_urls(result['images']['rb_checks'], task)
    template = get_template('report.jinja2')
    text = template.render(result=result, keys_scheme=keys_scheme, timedelta=datetime.timedelta, int=int)
    return result, text


def make_html_report(data, buildtype, from_date, to_date, stream):
    stats = stat_maker.calc_stat_for_buildtype_and_period(data, buildtype, from_date, to_date)
    result = {'stat': {'distbuild_linux': {'emulation': stats}}}
    keys_scheme = get_keys_scheme(result['stat'])
    template = get_template('report.jinja2')
    text = template.render(result=result, keys_scheme=keys_scheme, timedelta=datetime.timedelta, int=int)
    stream.write(text)


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG)
    result, text = make_report(datetime.timedelta(days=65), 'dump.csv')
    with open('text.html', 'w') as stream:
        stream.write(text)
    with open('result.json', 'w') as stream:
        json.dump(result, stream, indent=4, sort_keys=True)
