# -*- coding: utf-8 -*-

import subprocess
import shlex

from os import chdir
from glob import glob
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxInfoParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.environments import MongodbEnvironment


class MongoHostParameter(SandboxStringParameter):
    name = 'mongo_host'
    description = 'Host with mongo db'
    default_value = 'cms-dbs01g.search.yandex.net'


class MongoDbParameter(SandboxStringParameter):
    name = 'mongo_db'
    description = 'Database name'
    default_value = 'heartbeat'


class MongoDbBackupTTL(SandboxStringParameter):
    name = 'backup_ttl'
    description = 'Time (in days) to store backup'
    default_value = '14'


class MongoCollectionsParameter(SandboxStringParameter):
    name = 'mongo_collections'
    description = 'Comma-separated collections, or "all"'
    default_value = 'instanceusageunused'


class MongoAdditionalParametr(SandboxStringParameter):
    name = 'mongo_additional'
    description = 'Additional parameters to mongo client'
    default_value = ''


class MongoDumpAdditionalParametr(SandboxStringParameter):
    name = 'mongo_dump_additional'
    description = 'Additional parameters to mongodump'
    default_value = ''


class MongoCompressResource(SandboxBoolParameter):
    name = 'compress_resource'
    description = 'Compress downloaded data'
    required = False


class CustomBlock(SandboxInfoParameter):
    description = 'Mongodb with authorization'


class MongodbUser(SandboxStringParameter):
    name = 'mongodb_user'
    description = 'Mongodb username'
    required = False


class MongodbPasswordKey(SandboxStringParameter):
    name = 'mongodb_password_key'
    description = 'Mongodb password key in Sandbox vault'
    required = False


class MongoWithAuth(SandboxBoolParameter):
    name = 'mongo_auth'
    description = 'Add mongo auth options'
    required = False


MongoWithAuth.sub_fields = {
    'true': [CustomBlock.name, MongodbUser.name, MongodbPasswordKey.name]
}


class BackupMongo(SandboxTask):
    """
    Забэкапить базу mongo в надежное место
    """

    type = 'BACKUP_MONGO'

    input_parameters = [MongoHostParameter,
                        MongoDbParameter,
                        MongoCollectionsParameter,
                        MongoAdditionalParametr,
                        MongoDumpAdditionalParametr,
                        MongoCompressResource,
                        MongoDbBackupTTL,
                        MongoWithAuth,
                        CustomBlock,
                        MongodbUser,
                        MongodbPasswordKey,
                        ]

    environment = (
        MongodbEnvironment(),
    )

    def get_dump_path(self):
        return self.abs_path('dump')

    def on_enqueue(self):
        self.ctx['dump_id'] = self.create_resource(self.descr, self.get_dump_path(),
                                                   resource_types.MONGO_DB_DUMP, arch='any',
                                                   attributes={'db': self.ctx[MongoDbParameter.name],
                                                               'ttl': self.ctx[MongoDbBackupTTL.name],
                                                               'from_host': self.ctx[MongoHostParameter.name]}).id

    def on_execute(self):
        backup_collections = self.ctx[MongoCollectionsParameter.name].split(',')

        if self.ctx[MongoWithAuth.name]:
            auth = ['-u', self.ctx[MongodbUser.name],
                    '-p', self.get_vault_data(self.owner,
                                              self.ctx[MongodbPasswordKey.name])
                    ]
        else:
            auth = []

        # get available collections and check backup collections exist
        additional_parametrs = shlex.split(self.ctx[MongoAdditionalParametr.name])
        arguments = ['mongo', '%s/%s' % (self.ctx[MongoHostParameter.name],
                    self.ctx[MongoDbParameter.name]), '--eval', 'rs.slaveOk(); db.getCollectionNames()',
                    '--quiet'] + auth + additional_parametrs
        p = run_process(arguments, stdout=subprocess.PIPE, log_prefix='mongo_list_collections', outputs_to_one_file=False)
        res = p.communicate()
        avail_collections = res[0].strip().split('\n')[-1].split(',')

        additional_mongodump_parametrs = shlex.split(self.ctx[MongoDumpAdditionalParametr.name])

        if backup_collections[0] == 'all':
            for collection in avail_collections:
                arguments = ['mongodump', '-h', self.ctx[MongoHostParameter.name], '-d',
                            self.ctx[MongoDbParameter.name], '-c', collection] + auth + additional_mongodump_parametrs
                run_process(arguments, log_prefix='mongo_dump_collection')
        else:
            notfound_collections = set(backup_collections) - set(avail_collections)
            if len(notfound_collections):
                raise Exception, "Collections %s not found" % (','.join(notfound_collections))
            for collection in backup_collections:
                arguments = ['mongodump', '-h', self.ctx[MongoHostParameter.name], '-d',
                            self.ctx[MongoDbParameter.name], '-c', collection] + auth + additional_mongodump_parametrs
                run_process(arguments, log_prefix='mongo_dump_collection')

        if self.ctx[MongoCompressResource.name]:
            chdir(self.get_dump_path())
            arguments = ['tar', '-k', '--remove-files', '-zcvf', 'mongodumpdb.tar.gz'] + glob("*")
            run_process(arguments, log_prefix='tar')


__Task__ = BackupMongo
