# -*- coding: utf-8 -*-

import subprocess
import shlex
import shutil
import json
import logging
import time

from os import chdir, path
from glob import glob
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxInfoParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.projects.common.environments import MongodbEnvironment


class MongoHostParameter(SandboxStringParameter):
    name = 'mongo_host'
    description = 'Host with mongo db'
    default_value = 'cms-dbs01g.search.yandex.net'


class MongoDbParameter(SandboxStringParameter):
    name = 'mongo_db'
    description = 'Comma-separated database names, or "all"'
    default_value = 'heartbeat'


class MongoDbBackupTTL(SandboxStringParameter):
    name = 'backup_ttl'
    description = 'Time (in days) to store backup'
    default_value = '14'


class MongoCollectionsParameter(SandboxStringParameter):
    name = 'mongo_collections'
    description = 'Comma-separated collections, or "all"'
    default_value = 'instanceusageunused'


class MongoAdditionalParametr(SandboxStringParameter):
    name = 'mongo_additional'
    description = 'Additional parameters to mongo client'
    default_value = ''


class MongoDumpAdditionalParametr(SandboxStringParameter):
    name = 'mongo_dump_additional'
    description = 'Additional parameters to mongodump'
    default_value = ''


class MongoCompressResource(SandboxBoolParameter):
    name = 'compress_resource'
    description = 'Compress downloaded data'
    required = False


class Service(SandboxStringParameter):
    name = 'service'
    description = 'Service name'
    required = False


class CustomBlock(SandboxInfoParameter):
    description = 'Mongodb with authorization'


class MongodbUser(SandboxStringParameter):
    name = 'mongodb_user'
    description = 'Mongodb username'
    required = False


class MongodbPasswordKey(SandboxStringParameter):
    name = 'mongodb_password_key'
    description = 'Mongodb password key in Sandbox vault'
    required = False


class MongoWithAuth(SandboxBoolParameter):
    name = 'mongo_auth'
    description = 'Add mongo auth options'
    required = False


MongoWithAuth.sub_fields = {
    'true': [CustomBlock.name, MongodbUser.name, MongodbPasswordKey.name]
}


class BackupMongoManyDatabases(SandboxTask):
    """
    Забэкапить много баз данных mongo в надежное место
    """

    type = 'BACKUP_MONGO_MANY_DATABASES'

    input_parameters = [MongoHostParameter,
                        MongoDbParameter,
                        MongoCollectionsParameter,
                        MongoAdditionalParametr,
                        MongoDumpAdditionalParametr,
                        MongoCompressResource,
                        MongoDbBackupTTL,
                        Service,
                        MongoWithAuth,
                        CustomBlock,
                        MongodbUser,
                        MongodbPasswordKey,
                        ]

    environment = (
        MongodbEnvironment(),
    )

    def get_dump_path(self):
        return self.abs_path('dump')

    def on_enqueue(self):
        self.ctx['dump_id'] = self.create_resource(self.descr, self.get_dump_path(),
                                                   resource_types.MONGO_DB_DUMP, arch='any',
                                                   attributes={'db': self.ctx[MongoDbParameter.name],
                                                               'ttl': self.ctx[MongoDbBackupTTL.name],
                                                               'from_host': self.ctx[MongoHostParameter.name],
                                                               'service': self.ctx[Service.name]}).id

    def get_databases(self, additional_parametrs):
        arguments = ['mongo', self.ctx[MongoHostParameter.name], '--eval', 'rs.slaveOk(); printjson(db.adminCommand("listDatabases"))',
                    '--quiet'] + additional_parametrs
        p = run_process(arguments, stdout=subprocess.PIPE, log_prefix='mongo_list_databases', outputs_to_one_file=False)
        res = p.communicate()
        databasesReply = json.loads(res[0])
        databases = [db['name'] for db in databasesReply['databases']]
        return databases

    def get_collections(self, database, additional_parametrs):
        arguments = ['mongo', '%s/%s' % (self.ctx[MongoHostParameter.name],
                    database), '--eval', 'rs.slaveOk(); db.getCollectionNames()',
                    '--quiet'] + additional_parametrs
        p = run_process(arguments, stdout=subprocess.PIPE, log_prefix='mongo_list_collections', outputs_to_one_file=False)
        res = p.communicate()
        avail_collections = res[0].strip().split('\n')[-1].split(',')
        return avail_collections

    # check backup collections exist
    def validate_collections(self, databases, backup_collections, additional_parametrs):
        for database in databases:
            avail_collections = self.get_collections(database, additional_parametrs)
            notfound_collections = set(backup_collections) - set(avail_collections)
            if notfound_collections:
                raise Exception('Collections %s not found' % ','.join(notfound_collections))

    def on_execute(self):
        backup_collections = self.ctx[MongoCollectionsParameter.name].split(',')
        databases = self.ctx[MongoDbParameter.name].split(',')

        if self.ctx[MongoWithAuth.name]:
            auth = ['-u', self.ctx[MongodbUser.name],
                    '-p', self.get_vault_data(self.owner,
                                              self.ctx[MongodbPasswordKey.name])
                    ]
        else:
            auth = []

        additional_parametrs = auth + shlex.split(self.ctx[MongoAdditionalParametr.name])
        additional_mongodump_parametrs = auth + shlex.split(self.ctx[MongoDumpAdditionalParametr.name])

        if databases[0] == 'all':  # Backup them all!
            logging.info('Retrieve all databases')
            databases = self.get_databases(additional_parametrs)

        logging.info('Dump databases: %s', databases)

        if backup_collections[0] != 'all':
            logging.info('Validate collections existing: ', backup_collections)
            self.validate_collections(databases, backup_collections, additional_parametrs)

        for database in databases:
            if backup_collections[0] == 'all':
                collections = self.get_collections(database, additional_parametrs)
            else:
                collections = backup_collections

            db_outdir = path.join(self.get_dump_path(), database)

            for attempt in xrange(5, -1, -1):
                logging.info('Backup database: %s attempts left: %d', database, attempt)
                if path.exists(db_outdir):
                    shutil.rmtree(db_outdir)

                try:
                    for collection in collections:
                        logging.info('Backup collection %s/%s', database, collection)
                        arguments = ['mongodump', '-h', self.ctx[MongoHostParameter.name], '-d',
                                    database, '-c', collection] + additional_mongodump_parametrs
                        run_process(arguments, log_prefix='mongo_dump_collection')
                    break
                except SandboxSubprocessError:
                    logging.exception('Error dumping %s' % database)
                    time.sleep(5)
                    if attempt == 0:
                        raise

        if self.ctx[MongoCompressResource.name]:
            chdir(self.get_dump_path())
            arguments = ['tar', '-k', '--remove-files', '-zcvf', 'mongodumpdb.tar.gz'] + glob("*")
            run_process(arguments, log_prefix='tar')


__Task__ = BackupMongoManyDatabases
