# -*- coding: utf-8 -*-

import os
import time
import logging

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.process import run_process

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import apihelpers


class NannyServiceName(SandboxStringParameter):
    name = 'nanny_service_name'
    description = 'Nanny service name with mongodb'
    required = True


class MongodbReplicasetName(SandboxStringParameter):
    name = 'mongodb_replicaset_name'
    description = 'Mongodb replica set name'
    required = False


class MongodbUser(SandboxStringParameter):
    name = 'mongodb_user'
    description = 'Mongodb username'
    required = False


class MongodbPasswordKey(SandboxStringParameter):
    name = 'mongodb_password_key'
    description = 'Mongodb password key in Sandbox vault'
    required = False


class NannyMongodb(ResourceSelector):
    name = 'nanny_mongodb_utils'
    description = 'Which mongodb should we use (if not specified, we will use a mongodump from the nanny service)'
    resource_type = resource_types.NANNY_MONGODB
    required = False


class NannyOauthVaultKey(SandboxStringParameter):
    name = 'nanny_oauth_token_vault_key'
    description = 'Nanny oauth token key in Sandbox vault'
    required = True
    default_value = 'nanny_robot_oauth_token'


class MongodbAuthDbName(SandboxStringParameter):
    name = 'nanny_oauth_db_name'
    description = 'Nanny auth DB name (by default - admin)'
    required = False
    default_value = 'admin'


class MongodbAuthMechanism(SandboxStringParameter):
    name = 'nanny_oauth_mechanism'
    description = 'Nanny auth mechanism'
    required = False


class BackupUsersAndRoles(SandboxBoolParameter):
    name = 'backup_users_and_roles'
    description = 'Backup roles and users or not'
    required = False
    default_value = False


class BackupNannyMongodb(SandboxTask):
    """
    Забекапить базу mongodb из сервиса няни.

    https://st.yandex-team.ru/SWAT-1701

    Бекапим всю монгу, без разделения на базы и коллекции, с oplog, с мастера репликасета
    Подробнее тут https://wiki.yandex-team.ru/jandekspoisk/sepe/nanny/mongodbbackups/
    """

    type = 'BACKUP_NANNY_MONGODB'

    input_parameters = [
        NannyServiceName, MongodbReplicasetName, MongodbUser, MongodbPasswordKey,
        NannyMongodb, NannyOauthVaultKey, MongodbAuthDbName, BackupUsersAndRoles
    ]

    def on_execute(self):
        nanny_service_name = self.ctx[NannyServiceName.name]
        mongodb_replicaset_name = self.ctx.get(MongodbReplicasetName.name, nanny_service_name)
        logging.info('Try to backup mongodb from Nanny service {}'.format(nanny_service_name))
        nanny_oauth_token = self.get_vault_data(self.owner, self.ctx['nanny_oauth_token_vault_key'])
        nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_oauth_token)
        backup_path = self.path('mongodb_dump')
        resources = nanny_client.get_service_resources(nanny_service_name)
        snapshot_id = resources['snapshot_id']
        logging.info('Work with snapshot {}, {}'.format(snapshot_id, resources['content']))

        nanny_mongo = self.ctx.get(NannyMongodb.name)
        if not nanny_mongo:
            # ищем монгу в ресурсах сервиса, если оно так не указано
            for sandbox_resource in resources['content'].get('sandbox_files', []):
                if sandbox_resource.get('resource_type') == str(resource_types.NANNY_MONGODB):
                    nanny_mongo = sandbox_resource['task_id']
            if not nanny_mongo:
                raise SandboxTaskFailureError('Cannot find mongo in {} nanny service'.format(nanny_service_name))

            logging.info('Download mongodb from task: {}'.format(nanny_mongo))
            nanny_mongo = apihelpers.list_task_resources(
                nanny_mongo, resource_type=str(resource_types.NANNY_MONGODB)
            )[0]

        mongodb_path = self.sync_resource(nanny_mongo)

        instances = nanny_client.get_service_current_instances(nanny_service_name)[u'result']
        instances = ','.join(['{}:{}'.format(i['hostname'], i['port']) for i in instances])
        hosts_parameter = '{}/'.format(mongodb_replicaset_name) + instances

        arguments = [
            os.path.join(mongodb_path, 'bin', 'mongodump'),
            # хосты и порты из няни
            '--host', hosts_parameter,
            # уточняем, куда бекапить
            '--out', backup_path,
            '--oplog'
        ]

        if self.ctx.get(BackupUsersAndRoles.name):
            logging.info('Add --dumpDbUsersAndRoles parameter')
            arguments.append('--dumpDbUsersAndRoles')

        # если нужно, добавляем параметры для аутентификации
        mongodb_user = self.ctx.get(MongodbUser.name)
        if mongodb_user:
            logging.info('Setup auth parameters')
            logging.info('Try to get password for mongodb user {}'.format(mongodb_user))
            mongodb_password = self.get_vault_data(self.owner, self.ctx[MongodbPasswordKey.name])
            mongodb_auth_db = self.ctx.get(MongodbAuthDbName.name, MongodbAuthDbName.default_value)
            mongodb_auth_mechanism = self.ctx.get(MongodbAuthMechanism.name)
            arguments.append('--username')
            arguments.append(mongodb_user)
            arguments.append('--password')
            arguments.append(mongodb_password)
            arguments.append('--authenticationDatabase')
            arguments.append(mongodb_auth_db)
            if mongodb_auth_mechanism:
                arguments.append('--authenticationMechanism')
                arguments.append(mongodb_auth_mechanism)

        run_process(arguments, log_prefix='run_mongodump')

        logging.info('Save mongodb data ad Sandbox resource.')
        self.create_resource(self.descr, backup_path, resource_types.MONGO_DB_DUMP, arch='any',
                             attributes={
                                 'nanny_service': nanny_service_name,
                                 'backup_timestamp': int(time.time())
                             })


__Task__ = BackupNannyMongodb
