# -*- coding: utf-8 -*-

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers

from sandbox.projects.NewsAutoreleasableTask import NewsAutoreleasableTask
from sandbox.projects.NewsAutoreleasableTask import Params as NewsAutoreleasableTaskParams

from sandbox import common
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder

import shutil
import logging
import os.path


SECRET_OWNER = "NEWS"
SECRET_TOKEN_SERVICES = "sandbox", "nanny", "staff"


def compare_file(diff_path, old_file_path, new_file_path):
    diff_file_path = os.path.join(diff_path, os.path.basename(old_file_path) + '.diff')
    diff_err_file_path = diff_file_path + '.err'

    os.system('diff -uw ' + old_file_path + ' ' + new_file_path + ' > ' + diff_file_path + ' 2>' + diff_err_file_path)

    if os.path.exists(diff_err_file_path):
        err_msg = open(diff_err_file_path, 'r').read()

        if err_msg:
            raise Exception(err_msg)

    if not os.path.exists(diff_file_path):
        return False

    return bool(open(diff_file_path, 'r').readline())


class BakeNewsReleaserDeployGraphs(NewsAutoreleasableTask):
    """"""

    type = "BAKE_NEWS_RELEASER_DEPLOY_GRAPHS"

    def set_tokens(self):
        tokens_dir = self.abs_path('tokens')
        make_folder(tokens_dir)

        for token_service in SECRET_TOKEN_SERVICES:
            secret_name = "RELEASER_{}_TOKEN".format(token_service.upper())

            try:
                token = self.get_vault_data(SECRET_OWNER, secret_name)
                print("READ {} TOKEN".format(token_service))
            except common.errors.VaultError as e:
                print(e)
                raise common.errors.TaskFailure("Couldn't get {} token from the Vault".format(token_service))

            with open(os.path.join(tokens_dir, token_service), 'w') as token_file:
                token_file.write(token)

    def do_execute(self):
        baker = self.sync_resource(apihelpers.get_last_resource_with_attrs(
            'NEWS_RELEASER_BAKE_DEPLOY_GRAPHS_EXECUTABLE',
            {'released': 'stable'},
            all_attrs=True
        ))

        true_baker_path = self.path(os.path.basename(baker))
        shutil.copyfile(baker, true_baker_path)
        os.chmod(true_baker_path, 0755)

        graphs_dir = self.sync_resource(apihelpers.get_last_resource_with_attrs(
            'NEWS_RELEASER_DEPLOY_GRAPHS',
            {'released': 'stable'},
            all_attrs=True
        ))

        workcopy_dir = self.abs_path('workcopy')
        make_folder(workcopy_dir)

        self.set_tokens()

        dest_dir = 'baked_deploy_graphs'
        dest_path = os.path.join(workcopy_dir, dest_dir)

        baker_args = ' '.join(['='.join([k, v]) for k, v in {
            '--dest': dest_path,
            '--graphs-dir': graphs_dir,
        }.iteritems()])

        run_process('{} {}'.format(true_baker_path, baker_args), shell=True)

        resource = self.create_resource(
            description=self.descr,
            resource_path='workcopy/' + dest_dir,
            resource_type=resource_types.NEWS_RELEASER_BAKED_DEPLOY_GRAPHS,
            arch=self.arch,
            attributes={}
        )

        self.mark_resource_ready(resource.id)

        if (NewsAutoreleasableTaskParams.ReleaseTo.name in self.ctx) and self.ctx[NewsAutoreleasableTaskParams.ReleaseTo.name]:
            old_resource = apihelpers.get_last_resource_with_attrs(
                'NEWS_RELEASER_BAKED_DEPLOY_GRAPHS',
                {'released': self.ctx[NewsAutoreleasableTaskParams.ReleaseTo.name]},
                all_attrs=True
            )

            new_data = {}

            for new_file_name in os.listdir(dest_path):
                new_file_path = os.path.join(dest_path, new_file_name)

                if not os.path.isfile(new_file_path):
                    continue

                new_data[new_file_name] = new_file_path

            if old_resource:
                old_resource_path = self.sync_resource(old_resource)

                diff_dir = 'diffs'
                diff_path = os.path.join(workcopy_dir, diff_dir)

                make_folder(diff_path)

                old_data = set()
                have_errors = False
                have_diff = False

                for old_file_name in os.listdir(old_resource_path):
                    old_file_path = os.path.join(old_resource_path, old_file_name)

                    if not os.path.isfile(old_file_path):
                        continue

                    old_data.add(old_file_name)

                    if old_file_name in new_data:
                        try:
                            if compare_file(diff_path, old_file_path, new_data[old_file_name]):
                                have_diff = True

                        except Exception as e:
                            logging.error(e)
                            have_errors = True

                    else:
                        have_diff = True

                        with open(os.path.join(diff_path, os.path.basename(old_file_path) + '.removed'), 'w') as fd:
                            fd.write('file removed\n')

                for new_file_name in new_data.keys():
                    if new_file_name not in old_data:
                        have_diff = True

                        with open(os.path.join(diff_path, os.path.basename(old_file_path) + '.new'), 'w') as fd:
                            fd.write('new file\n')

                if have_diff or have_errors:
                    diff_resource_attrs = {
                        'previous_resource': old_resource.id
                    }

                    diff_resource = self.create_resource(self.descr, 'workcopy/' + diff_dir, resource_types.NEWS_RELEASER_BAKED_DEPLOY_GRAPHS_DIFF, arch='any', attributes=diff_resource_attrs)
                    self.mark_resource_ready(diff_resource.id)

                if have_errors:
                    raise Exception("Something went wrong")

                return have_diff

        return True


__Task__ = BakeNewsReleaserDeployGraphs
