# -*- coding: utf-8 -*-
import logging
import os

from sandbox.projects import resource_types
from sandbox.projects.common.wizard import printwizard as pw
from sandbox.projects.common.wizard import utils as wizard_utils
from sandbox.projects.websearch.begemot import resources as br
from sandbox.projects.websearch.begemot.common.fast_build import ShardSyncHelper
from sandbox.projects.websearch.begemot.parameters_sdk1 import BinaryParameter, ConfigParameter, PlanParameter, FastBuildConfigParameter, FreshParameter
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel

NANNY_BEGEMOT_ID = 'wizard_begemot_production_merger'


class ResponsesMode(sp.SandboxStringParameter):
    name = 'mode'
    description = 'Get Begemot responses or diff config'
    choices = [
        ('cgi requests -> responses', 'cgi'),
        ('apphost json -> responses', 'apphost'),
        ('anything -> rule schema for the shard', 'diff_conf')
    ]
    sub_fields = {
        'cgi': [PlanParameter.name],
        'apphost': [PlanParameter.name]
    }
    default_value = 'cgi'


class CacheSize(sp.SandboxIntegerParameter):
    name = 'cache_size'
    description = 'Begemot cache size in bytes'
    default_value = 0


class BegemotRunLightTests(task.SandboxTask):

    type = 'BEGEMOT_RUN_LIGHT_TESTS'
    execution_space = 42 * 1024

    input_parameters = (
        BinaryParameter,
        ResponsesMode,
        ConfigParameter,
        FastBuildConfigParameter,
        FreshParameter,
        PlanParameter,
        CacheSize
    )

    client_tags = wizard_utils.ALL_SANDBOX_HOSTS_TAGS

    def on_enqueue(self):
        self.create_resource(
            resource_type=br.BEGEMOT_RESPONSES_RESULT,
            description=self.description + ' output',
            resource_path='output.txt',
        )
        self.create_resource(
            resource_type=resource_types.PLAIN_TEXT_QUERIES,
            description=self.description + ' err',
            resource_path='err.txt',
        )
        self.create_resource(
            resource_type=br.BEGEMOT_EVLOG,
            description=self.description + ' eventlog',
            resource_path='bg_eventlog',
        )
        wizard_utils.setup_hosts_sdk1(self)
        wizard_utils.on_enqueue(task)

    def on_execute(self):
        os.environ["MKL_CBWR"] = "COMPATIBLE"
        begemot_binary = self.sync_resource(self.ctx[BinaryParameter.name])
        resources_out = channel.sandbox.list_resources(resource_type=br.BEGEMOT_RESPONSES_RESULT, task_id=self.id)
        resources_err = channel.sandbox.list_resources(resource_type=resource_types.PLAIN_TEXT_QUERIES, task_id=self.id)
        resources_evlog = channel.sandbox.list_resources(resource_type=br.BEGEMOT_EVLOG, task_id=self.id)
        output = resources_out[0].path
        err = resources_err[0].path
        evlog = resources_evlog[0].path

        shard_helper = ShardSyncHelper(self.ctx.get(FastBuildConfigParameter.name), sdk1_task=self)
        args = [begemot_binary, '--data', shard_helper.sync_shard()]

        if self.ctx.get(FreshParameter.name):
            args.extend(['--fresh', self.sync_resource(self.ctx[FreshParameter.name])])

        if self.ctx.get(ResponsesMode.name) == 'diff_conf':
            args.append('--print-bgschema')
            with open(output, 'w') as output_file, open(err, 'w') as err_file:
                process.run_process(args, wait=True, stdout=output_file, stderr=err_file)
            open(evlog, 'a').close()
        else:
            args.extend([
                '--cfg', os.path.join(self.sync_resource(self.ctx[ConfigParameter.name]), 'test/worker.cfg'),
                '--test',
                '--log', evlog,
                '--cache_size', str(self.ctx.get(CacheSize.name)),
            ])
            input = self.sync_resource(self.ctx[PlanParameter.name])
            if self.ctx.get(ResponsesMode.name) == 'cgi':
                args.append('--cgi')
                if channel.sandbox.get_resource(self.ctx[PlanParameter.name]).type != str(br.BEGEMOT_CGI_PLAN):
                    requests = input
                    input = self.create_resource(
                        'Prepared requests for begemot',
                        os.path.join(self.abs_path(), 'begemot_plan.txt'),
                        resource_type=br.BEGEMOT_CGI_PLAN
                    ).path
                    with open(input, 'w') as out:
                        for line in pw.read_wizard_requests(requests):
                            out.write(pw.parse_query(line)[0].partition('?')[2] + '\n')
            logging.info('input=' + input)
            logging.info('output=' + output)
            logging.info('err=' + err)
            with open(input, 'r') as input_file, open(output, 'w') as output_file, open(err, 'w') as err_file:
                process.run_process(args, wait=True, stdin=input_file, stdout=output_file, stderr=err_file)


__Task__ = BegemotRunLightTests
