# -*- coding: utf-8 -*-
import logging
import os
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import profiling
from sandbox.projects.common.apihelpers import get_last_released_resource
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import performance
from sandbox.projects.common.search.components.apphost import AppHost
from sandbox.projects.common.wizard import utils as wizard_utils
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot import resources as br
from sandbox.projects.websearch.begemot.common.fast_build import ShardSyncHelper
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import task

APPHOST_PORT = 8870


class BegemotBinary(sp.ResourceSelector):
    name = 'begemot_binary'
    description = 'Begemot binary'
    resource_type = br.BEGEMOT_EXECUTABLE
    required = True


class BegemotConfig(sp.ResourceSelector):
    name = 'begemot_config'
    description = 'Begemot config'
    resource_type = br.BEGEMOT_CONFIG
    required = True


class FastBuildConfig(sp.ResourceSelector):
    name = 'fast_build_config'
    description = 'Fast build config for begemot shard'
    resource_type = [shard.fast_build_config_resource_type for shard in AllBegemotServices.Service.itervalues()]
    required = False


class BegemotPlan(sp.ResourceSelector):
    name = 'begemot_plan'
    description = 'Begemot plan'
    resource_type = br.BEGEMOT_APP_HOST_PLAN
    required = True


class BegemotCache(sp.SandboxIntegerParameter):
    name = 'begemot_cache'
    description = 'Begemot cache size (bytes)'


class BegemotJobs(sp.SandboxIntegerParameter):
    name = 'begemot_jobs'
    description = 'Begemot jobs count'


class BegemotProtobuf(sp.SandboxBoolParameter):
    name = 'begemot_protobuf'
    description = 'Use protobuf (cross-worker) protocol'


class BegemotTestPerformance(performance.OldShootingTask, task.SandboxTask, profiling.ProfilingTask):
    type = 'BEGEMOT_TEST_PERFORMANCE'
    required_ram = 100 * 1024
    execution_space = 100 * 1024
    client_tags = wizard_utils.ALL_SANDBOX_HOSTS_TAGS & wizard_utils.BEGEMOT_PERF_HARDWARE_TAGS

    input_parameters = (
        BegemotBinary,
        BegemotConfig,
        FastBuildConfig,
        BegemotPlan,
        BegemotCache,
        BegemotJobs,
        BegemotProtobuf,
    ) + performance.OldShootingTask.shoot_input_parameters + profiling.ProfilingTask.input_parameters

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        wizard_utils.setup_hosts_sdk1(self, additional_restrictions=wizard_utils.BEGEMOT_PERF_HARDWARE_TAGS)
        wizard_utils.on_enqueue(self)

    def on_execute(self):
        begemot_binary_path = self.sync_resource(self.ctx[BegemotBinary.name])
        begemot_config_path = self.sync_resource(self.ctx[BegemotConfig.name])

        shard_helper = ShardSyncHelper(self.ctx.get(FastBuildConfig.name), sdk1_task=self)
        begemot_data_path = shard_helper.sync_shard()

        self.create_resource('Begemot eventlog', 'begemot.evlog', resource_types.OTHER_RESOURCE)
        begemot = search_components.get_begemot(
            begemot_binary_path, begemot_config_path,
            worker_dir=begemot_data_path, eventlog_path='begemot.evlog',
            cache_size=self.ctx.get(BegemotCache.name, 0),
            jobs_count=self.ctx.get(BegemotJobs.name, None),
        )
        # Prepare apphost
        apphost_resource = get_last_released_resource(resource_types.APP_HOST_BUNDLE)
        apphost_resource_path = self.sync_resource(apphost_resource.id)
        apphost_path = os.path.join(self.abs_path(), 'apphost_files')
        os.mkdir(apphost_path)
        apphost_path = self._unpack_apphost(apphost_resource_path, apphost_path)
        self._create_apphost_graph(apphost_path)
        app_host_config_path = self._create_apphost_config(apphost_path)
        logging.debug('binary path is %s', os.path.join(apphost_path, 'app_host'))
        apphost = AppHost(self, os.path.join(apphost_path, 'app_host'), app_host_config_path, port=APPHOST_PORT)

        self._profiling_init(begemot, self.__get_perf_data_path())
        self._init_virtualenv()
        with begemot, apphost:
            self._old_shoot(apphost, self.ctx[BegemotPlan.name])

        self._profiling_report(begemot, self.__get_perf_data_path())

    def get_short_task_result(self):
        if self.is_completed() and "max_rps" in self.ctx:
            return "{:0.2f}".format(self.ctx["max_rps"])

    def __get_perf_data_path(self, recreate=False):
        return self.abs_path("perf.data")

    @staticmethod
    def _unpack_apphost(apphost_resource_path, dest_path):
        with tarfile.open(apphost_resource_path, 'r:gz') as tar:
            tar.extractall(dest_path, tar.getmembers())
        return os.path.join(dest_path, os.path.basename(apphost_resource_path.replace('.tar.gz', '')))

    def _create_apphost_graph(self, apphost_path, begemot_port=search_components.DEFAULT_BEGEMOT_PORT):
        graph_file_content = {
            'name': 'begemot',
            'graph': {
                'BEGEMOT': ['BEGEMOT_CONFIG', 'INIT@request,report,user,flags,settings,merger'],
                'RESPONSE': ['BEGEMOT']
            },
            'version': 2,
            'sources': {
                'BEGEMOT_CONFIG': {
                    'codecs': ['lz4'],
                    'embed': [{'type': 'begemot_config'}]
                },
                'BEGEMOT': {
                    'codecs': ['lz4'],
                    'backend_descrs': [{'addr': 'netliba://localhost:{}'.format(begemot_port)}],
                    'timeout': 150
                },
            }
        }
        for source in graph_file_content['sources'].values():
            if 'backend_descrs' in source:
                source['backend_config'] = {'backend_descrs': source['backend_descrs']}
        if self.ctx.get(BegemotProtobuf.name):
            graph_file_content['sources']['BEGEMOT_CONFIG']['embed'][0]['binary'] = True
        os.mkdir(os.path.join(apphost_path, 'graphs'))
        fu.json_dump(os.path.join(apphost_path, 'graphs', 'begemot.json'), graph_file_content, indent=4)

    @staticmethod
    def _create_apphost_config(apphost_path, apphost_port=APPHOST_PORT):
        config_file_content = {
            'port': apphost_port,
            'executor_threads': 1,
            'threads': 2,
            'engine': {'async': 1},
            'total_quota': 1000,
            'group_quotas': {},
            'conf_dir': os.path.join(apphost_path, 'graphs'),
            'fallback_conf_dir': os.path.join(apphost_path, 'graphs'),
            'protocol_options': {'post/ConnectTimeout': '25ms'},
            'log': 'apphost.evlog',
            'default_load_control_config': {},
            'update_from_fs_config': {}
        }
        apphost_cfg_path = os.path.join(apphost_path, 'apphost.cfg')
        fu.json_dump(apphost_cfg_path, config_file_content, indent=4)
        return apphost_cfg_path


__Task__ = BegemotTestPerformance
