# -*- coding: utf-8 -*-

import json
import re
import requests
from os import path
import logging
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common import utils
from sandbox.projects.common.blender.fml_config_rpc_client import FmlConfigClient as Client
from sandbox.projects.common.blender.task_params import RearrReeourceId, FmlConfigServerHostParameter


betas = {
    'ru': {'host': 'hamster.yandex.ru', 'lr': 213},
    'by': {'host': 'hamster.yandex.by', 'lr': 157},
    'ua': {'host': 'hamster.yandex.ua', 'lr': 143},
    'kz': {'host': 'hamster.yandex.kz', 'lr': 163},
    'tr': {'host': 'hamster.yandex.com.tr', 'lr': 983},
}


def _extract_fml_from_rearr(rearr_param):
    reg_exp = re.compile('.*"fml":"(.*)"')
    match = reg_exp.match(rearr_param)
    if not match:
        raise SandboxTaskFailureError('Wrong rearr parameter')

    return match.group(1)


def _check_formula_works(host, rearr_param, fml_name, query, lr):
    req_text = 'https://' + host + '/search/?text=' + query + '&rearr=' + rearr_param + '&lr=' + str(lr) + \
        '&json_dump=search_props.UPPER.0.properties&no-tests=da&nocache=da&exp_flags=disable_https&waitall=da&noredirect=1&timeout=10000000&sbh=1'

    logging.info('req ' + req_text)
    response = requests.get(req_text, verify=False).json()

    fmls = response.get('search_props.UPPER.0.properties', {}).get('ApplyBlender.fmls', '')
    logging.info('fmls ' + fmls)
    if fml_name in fmls:
        return True

    return False


class BlenderCheckRearrSet(SandboxTask):
    """
        Таск для проверки набора параметров, создааных для обратного блендерного эксперимаента.

        * Для каждой новой формулы проверяет наличие ее в транке и в последнем rd релизе.
        * Для ряда вертикалей и локалей проверяет что формула включилась в продакшене.

        Страница на wiki:
        https://wiki.yandex-team.ru/users/timofeevay/backexperiments/
    """
    type = 'BLENDER_CHECK_REARR_SET'

    input_parameters = [
        RearrReeourceId,
        FmlConfigServerHostParameter,
    ]

    def on_execute(self):
        client = Client(self.ctx['fml_config_server_host'], '')

        rearr_params_file_name = utils.sync_resource(self.ctx['rearr_resource_id'])

        rearr_params = []
        fmls_to_check = []
        logging.info('Rearr set file name: ' + rearr_params_file_name)
        with open(rearr_params_file_name) as rearr_params_file:
            experiment_params = json.load(rearr_params_file)
            rearr_params = experiment_params['rearr']
            fmls_to_check = experiment_params['new_formulas']

        for fml_name in fmls_to_check:
            if fml_name == 'stub_fml':
                continue

            logging.info('Checking of the formula %s is started' % fml_name)

            formula_check = client.check_formula(fml_name, 'trunk')
            if not formula_check['exists']:
                raise SandboxTaskFailureError('Formula %s does not exist in the trunk' % fml_name)

            releases = client.get_rd_releases()
            cur_release = releases[0]
            cur_branch = client.get_rd_release_info(cur_release['sandbox_task'])['svn_branch']

            formula_check = client.check_formula(fml_name, cur_branch)
            if not formula_check['exists']:
                raise SandboxTaskFailureError('Formula %s does not exist in the last release' % fml_name)

        queries = {}
        queries_file_path = path.join(path.dirname(__file__), 'queries.json')
        logging.info('Open ' + queries_file_path)
        with open(queries_file_path) as queries_file:
            queries = json.load(queries_file)

        for rearr_param in rearr_params:
            fml_name = _extract_fml_from_rearr(rearr_param)
            locale = rearr_param.split('/')[4].strip('"')
            vertical = rearr_param.split('/')[3]
            condition = rearr_param.split('/')[5].strip('"')
            if condition != '1':
                continue

            if fml_name == 'stub_fml':
                continue

            host = betas.get(locale, {}).get('host', '')
            lr = betas.get(locale, {}).get('lr', 225)
            query = queries.get(vertical, {}).get(locale, '')
            if host and query:
                logging.info('fml: %s, host: %s, rearr: %s, query: %s' % (fml_name, host, rearr_param, query))
                if not _check_formula_works(host, rearr_param, fml_name, query, lr):
                    raise SandboxTaskFailureError('Formula %s does not work on beta' % fml_name)

        logging.info('Check is finished')


__Task__ = BlenderCheckRearrSet
