# -*- coding: utf-8 -*-OF

import base64
import datetime
import imp
import json
import logging
import os
import shutil
import subprocess
import tarfile
import tempfile
import time
import sandbox.projects.common.constants as consts
import sandbox.projects.common.blender.settings as settings
import sandbox.projects.resource_types as resource_types
import sandbox.sandboxsdk.ssh as ssh
from collections import defaultdict
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.common.build import ArcadiaTask
from sandbox.projects.common.arcadia import sdk
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.blender.fml_config_rpc_client import FmlConfigClient as Client
from sandbox.projects.common.blender.task_params import DateParameter, UserNameParameter, FmlConfigServerHostParameter, ArcadiaPathParameter, RDSvnPathParameter


FAKE_FMLS_IN_NOA = ['music', 'video_blend', 'auto_filter', 'music_filter']


class DoUploadParam(sp.SandboxBoolParameter):
    name = "do_upload"
    description = "Upload formulas"
    default_value = True


class DoCommitParam(sp.SandboxBoolParameter):
    name = "do_commit"
    description = "Commit changes"
    default_value = True


class CanonizeFmlConfig(sp.SandboxBoolParameter):
    name = "canonize_fml_config"
    description = "Canonize output fml_config parameters"
    default_value = True


def assign_to_path(dic, path, value):
    path = [eval(p) if len(p) and p[0] == '"' else p for p in path.split('/')]
    for p in path[0:-1]:
        if p not in dic:
            dic[p] = {}
        dic = dic[p]
    dic[path[-1]] = value


class BlenderCreateBackExperiment(ArcadiaTask.ArcadiaTask):
    """
        Таск для создания обратного блендерного эксперимента.

        * Получает набор формул, актуальных на заданную дату.
        * Заново коммитит их в репозиторий.
        * Создает ресурс вида BLENDER_EXPERIMENT_SETUP с набором rearr параметров для включения эксперимента.

        Страница на wiki:
        https://wiki.yandex-team.ru/users/timofeevay/backexperiments/
    """
    type = 'BLENDER_CREATE_BACK_EXPERIMENT'

    input_parameters = [
        DateParameter,
        UserNameParameter,
        FmlConfigServerHostParameter,
        ArcadiaPathParameter,
        RDSvnPathParameter,
        DoUploadParam,
        DoCommitParam,
        CanonizeFmlConfig,
    ]

    def _get_release_formulas_info(self, client, release_branch):
        release_verticals = client.get_verticals(release_branch)
        release_locales = ['ru', 'ua', 'by', 'kz', 'tr']
        uis = ['desktop', 'pad', 'touch', 'ystroka']

        rearr_dict = defaultdict(lambda: defaultdict(lambda: defaultdict()))
        new_formulas = set()

        for vertical in release_verticals:
            for ui in uis:
                for locale in release_locales:
                    formulas = client.get_prod(vertical, ui, locale, release_branch, all_formulas=True, get_config_node=True)
                    if not formulas[locale] or not formulas[locale]['fmls']:
                        continue

                    fmls_to_replace = {}
                    for fml_key, fml_name in formulas[locale]['fmls'].items():
                        if fml_name in FAKE_FMLS_IN_NOA:
                            continue
                        fml_names = [fml_name + '_' + locale, fml_name]
                        checked_fmls = [name for name in fml_names if client.check_formula(name, release_branch)['exists']]
                        if not checked_fmls:
                            raise SandboxTaskFailureError('Can not find formula ' + fml_name)
                        fmls_to_replace[fml_key] = checked_fmls[0]

                    for fml_key, fml_name in fmls_to_replace.items():
                        formulas[locale]['fmls'][fml_key] = fml_name
                        formulas[locale]['config'][fml_key] = fml_name

                    rearr_dict[vertical][ui][locale] = formulas[locale]
                    for formula in formulas[locale]['fmls'].values():
                        if formula not in FAKE_FMLS_IN_NOA:
                            new_formulas.add(formula)

        new_formulas.discard('stub_fml')

        return new_formulas, rearr_dict

    def _upload_formula(self, formula, upload_fml_path):
        imp.load_source('upload_fml_module', upload_fml_path)
        from upload_fml_module import UPLOAD_FML_SETTINGS, get_cmake_lists_for_formula, update_cmake_lists

        _, formula_name = os.path.split(formula)

        resource_path = formula
        if UPLOAD_FML_SETTINGS.PACK_TO_TAR:
            resource_path = self.path(formula_name + '.tar.gz')
            with tarfile.open(resource_path, 'w:gz') as tar:
                tar.add(formula, formula_name)

        resource_type = getattr(resource_types, UPLOAD_FML_SETTINGS.FML_RESOURCE_TYPE)
        if self.ctx.get(DoUploadParam.name):
            resource = self.create_resource(
                description='Back experiment formula',
                resource_path=resource_path,
                resource_type=resource_type,
                attributes={'ttl': UPLOAD_FML_SETTINGS.TTL},
            )
            self.mark_resource_ready(resource.id)
            resource_id = resource.id
        else:
            logging.info("Skipping uploading for " + formula)
            resource_id = 345345345
        cmake_path = get_cmake_lists_for_formula(formula)
        update_cmake_lists(formula_name, str(resource_id), cmake_path, False)

    def _canonize_fml_config(self, patch_fml_config_path, contents):
        f, fn = tempfile.mkstemp()
        try:
            os.close(f)
            with open(fn, 'w') as f:
                json.dump(contents, f)
            subprocess.call([patch_fml_config_path, 'canon', '-c', fn])
            with open(fn, 'r') as f:
                res = json.load(f)
        finally:
            os.remove(fn)
        return res

    def on_enqueue(self):
        ArcadiaTask.ArcadiaTask.on_enqueue(self)
        channel.task = self
        self.ctx['out_resource_id'] = self.create_resource(
            self.descr,
            'rearrange_params.json',
            resource_types.BLENDER_EXPERIMENT_SETUP,
            arch='any',
            attributes={
                "ttl": "inf",
            }
        ).id
        self.ctx[consts.ARCADIA_URL_KEY] = self.ctx['arcadia_path']

    def on_execute(self):
        client = Client(self.ctx['fml_config_server_host'], '')
        releases = client.get_rd_releases()

        date = datetime.datetime.strptime(self.ctx['date'], '%d/%m/%Y')
        timestamp = int(time.mktime(date.timetuple())) * 1000

        release_is_found = False
        for release in releases:
            if release['creation_time'] < timestamp:
                release_is_found = True
                break

        if not release_is_found:
            raise SandboxTaskFailureError('Can not find release for date ' + str(date))

        release_branch = client.get_rd_release_info(release['sandbox_task'])['svn_branch']
        release_name = release_branch.split('/')[-1]

        logging.info('Restoring formulas for release: ' + release_name)
        logging.info('Gathering formulas...')

        new_formulas, rearr_dict = self._get_release_formulas_info(client, release_branch)
        logging.info('New formulas: ' + str(new_formulas))
        logging.info('Rearr dict: ' + str(rearr_dict))

        logging.info('Uploading formulas...')
        arcadia_src_dir = self.get_arcadia_src_dir()
        Arcadia.checkout(self.ctx['rd_svn_path'], self.path(settings.RD_LOCAL_PATH))

        upload_fml_path = os.path.join(arcadia_src_dir, settings.RD_PATH, 'upload_fml.py')
        patch_fml_config_path = os.path.join(arcadia_src_dir, settings.PATCH_FML_CONFIG_PATH, 'patch_fml_config.py')

        uploaded_fmls = set()
        formula_name_to_new = {}
        for fml_name in new_formulas:
            new_formula = 'old_blndr_' + release_name + '_' + fml_name
            new_formula_check = client.check_formula(new_formula, 'trunk')
            formula_name_to_new[fml_name] = new_formula

            if not new_formula_check['exists']:
                logging.info('Get formula info: ' + fml_name + '\t' + release_branch)
                fml = client.get_formula(fml_name, release_branch)
                if 'b64' not in fml:
                    raise SandboxTaskFailureError('Can not find formula ' + fml_name)
                raw_fml = base64.b64decode(fml['b64'])
                logging.info('To upload formula ' + fml_name)
                # uploading formula to trunk
                new_fml_file_name = self.path(os.path.join(settings.RD_LOCAL_PATH, 'blender', 'old_blndr_' + release_name + '_' + fml['fname']))
                with open(new_fml_file_name, 'wb') as f:
                    f.write(raw_fml)

                self._upload_formula(new_fml_file_name, upload_fml_path)
                uploaded_fmls.add(new_formula)

        blender_cmake_file_path = self.path(os.path.join(settings.RD_LOCAL_PATH, settings.RD_BLENDER_CMAKE_PATH))
        shutil.copyfile(blender_cmake_file_path, os.path.join(arcadia_src_dir, settings.RD_PATH, settings.RD_BLENDER_CMAKE_PATH))
        if self.ctx.get(DoUploadParam.name):
            sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                arcadia_src_dir,
                ["search/web/rearrs_upper"],
                consts.RELEASE_BUILD_TYPE,
                results_dir=self.path(settings.RD_BUILD_PATH),
                clear_build=False,
                test=True,
            )

        commit_msg = '[diff-resolver:' + self.ctx['user_name'] + ']' + ' Creating blender back experiment. Sandbox task id: ' + str(self.id)
        with ssh.Key(self, settings.ROBOT_KEY_OWNER, settings.ROBOT_KEY_NAME):
            if self.ctx.get(DoUploadParam.name) and self.ctx.get(DoCommitParam.name):
                Arcadia.commit(blender_cmake_file_path, commit_msg, settings.ROBOT_NAME)
            else:
                logging.info("Commit message: " + commit_msg)
#                logging.info("Diff:\n" + Arcadia.revision_diff(blender_cmake_file_path, 'HEAD'))  # Does not work

        rearr_params = []
        collected_rearr = {}
        rearr_by_ui = defaultdict(lambda: defaultdict())
        for vertical, v_params in rearr_dict.items():
            for ui, v_ui_params in v_params.items():
                for locale, params in v_ui_params.items():
                    prod_path = client.get_prod_path(vertical, ui, locale, 'trunk')['path']
                    config = params['config']
                    for fml_key, fml_name in params['fmls'].items():
                        if fml_name in formula_name_to_new:
                            config[fml_key] = formula_name_to_new[fml_name]
                    rearr_params.append(('scheme_Local/BlenderFmls/FmlConfig/' + prod_path + '=' + json.dumps(config)).replace(' ', ''))
                    assign_to_path(collected_rearr, prod_path, config)
                    assign_to_path(rearr_by_ui[ui], prod_path, config)
        if self.ctx.get(CanonizeFmlConfig.name):
            collected_rearr = self._canonize_fml_config(patch_fml_config_path, collected_rearr)
            for ui in rearr_by_ui:
                rearr_by_ui[ui] = self._canonize_fml_config(patch_fml_config_path, rearr_by_ui[ui])
        collected_rearr = 'scheme_Local/BlenderFmls/FmlConfig=' + json.dumps(collected_rearr).replace(' ', '')
        for ui in rearr_by_ui:
            rearr_by_ui[ui] = 'scheme_Local/BlenderFmls/FmlConfig=' + json.dumps(rearr_by_ui[ui]).replace(' ', '')

        out_res = channel.sandbox.get_resource(self.ctx['out_resource_id'])
        result = {}
        result['rearrs'] = rearr_params
        result['collected_rearr'] = collected_rearr
        for ui in rearr_by_ui:
            result['rearr_for_' + ui] = rearr_by_ui[ui]
        result['uploaded_formulas'] = list(sorted(uploaded_fmls))
        result['new_formulas'] = formula_name_to_new.values()
        with open(out_res.path, 'w') as fout:
            fout.write(json.dumps(result, indent=4, sort_keys=True, ensure_ascii=False))

        self.ctx['rearrs'] = rearr_params
        self.ctx['collected_rearr'] = collected_rearr
        for ui in rearr_by_ui:
            self.ctx['rearr_for_' + ui] = rearr_by_ui[ui]
        self.ctx['uploaded_formulas'] = list(sorted(uploaded_fmls))
        self.ctx['new_formulas'] = formula_name_to_new.values()
        logging.info('All formulas are restored')


__Task__ = BlenderCreateBackExperiment
