# -*- coding: utf-8 -*-

import json
import shutil
from os import path
import logging
import sandbox.sandboxsdk.ssh as ssh
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common import utils
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import ArcadiaTask
import sandbox.projects.common.constants as consts
import sandbox.projects.common.blender.settings as settings
from sandbox.projects.common.blender.fml_config_rpc_client import FmlConfigClient as Client
from sandbox.projects.common.blender.task_params import RearrReeourceId, UserNameParameter, FmlConfigServerHostParameter, ArcadiaPathParameter, RDSvnPathParameter, PatchFmlConfigSvnParameter


class BlenderRemoveExperiment(ArcadiaTask.ArcadiaTask):
    """
        Таск для удаления обратного блендерного эксперимаента.

        * Удаляет формулы обратного эксперимента из r.d

        Страница на wiki:
        https://wiki.yandex-team.ru/users/timofeevay/backexperiments/
    """
    type = 'BLENDER_REMOVE_EXPERIMENT'

    input_parameters = [
        RearrReeourceId,
        UserNameParameter,
        FmlConfigServerHostParameter,
        ArcadiaPathParameter,
        RDSvnPathParameter,
        PatchFmlConfigSvnParameter,
    ]

    def on_enqueue(self):
        ArcadiaTask.ArcadiaTask.on_enqueue(self)
        self.ctx[consts.ARCADIA_URL_KEY] = self.ctx['arcadia_path']

    def on_execute(self):
        client = Client(self.ctx['fml_config_server_host'], '')

        rearr_params_file_name = utils.sync_resource(self.ctx['rearr_resource_id'])

        fmls_to_remove = []
        logging.info('Rearr set file name: ' + rearr_params_file_name)
        with open(rearr_params_file_name) as rearr_params_file:
            fmls_to_remove = json.load(rearr_params_file)['new_formulas']

        rd_local_path = self.path(settings.RD_LOCAL_PATH)
        patch_fml_local = self.path(settings.PATCH_FML_CONFIG_LOCAL_PATH)

        Arcadia.checkout(self.ctx['rd_svn_path'], rd_local_path)
        Arcadia.checkout(self.ctx['patch_fml_config_svn_path'], patch_fml_local)

        eraser_params = [
            path.join(rd_local_path, 'rm_fml.py'),
            '--pfm',
            path.join(patch_fml_local, 'patch_fml_config.py')]

        to_erase = False

        for fml_name in fmls_to_remove:
            logging.info('Removing of the formula ' + fml_name)

            formula_check = client.check_formula(fml_name, 'trunk')
            if not formula_check['exists']:
                logging.info('Formula %s does not exist in trunk' % fml_name)
                continue

            eraser_params.append(fml_name)
            to_erase = True

        blender_cmake_file_path = self.path(path.join(settings.RD_LOCAL_PATH, settings.RD_BLENDER_CMAKE_PATH))
        if to_erase:
            process.run_process(eraser_params)

            arcadia_src_dir = self.get_arcadia_src_dir()

            logging.info('Copying ya.make to arcadia dir')
            shutil.copyfile(blender_cmake_file_path, path.join(arcadia_src_dir, settings.RD_PATH, settings.RD_BLENDER_CMAKE_PATH))
            sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                arcadia_src_dir,
                ["search/web/rearrs_upper"],
                consts.RELEASE_BUILD_TYPE,
                results_dir=settings.RD_BUILD_PATH,
                clear_build=False,
                test=True,
            )
            commit_msg = 'Removing of the blender back experiment. Sandbox task id: ' + str(self.id) + ', committer: ' + self.ctx['user_name']
            with ssh.Key(self, settings.ROBOT_KEY_OWNER, settings.ROBOT_KEY_NAME):
                Arcadia.commit(blender_cmake_file_path, commit_msg, settings.ROBOT_NAME)

        logging.info('Removing is finished')


__Task__ = BlenderRemoveExperiment
