import codecs
import json
import urlparse
import os.path

from sandbox.projects.common.apihelpers import get_last_resource
from sandbox.projects.common.bno.resources import save_resource
from sandbox.projects.common.bno.task import BnoBaseBuildTask, VpsHost
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.BnoAppHostToAppBuild.resource_types import BNO_URL_TO_APP
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxRadioParameter, SandboxBoolParameter


def parse_gplay(url):
    if not url:
        return None
    result = urlparse.parse_qs(urlparse.urlparse(url).query)
    try:
        return {'id': result['id'][0].strip('#').strip(), 'country': result['gl'][0]}
    except KeyError:
        return None


def parse_istore(url):
    url = '' if url is None else url
    result = urlparse.urlparse(url).path.split('/')[1:]
    return {'id': result[2].strip(), 'country': result[0]} if len(result) == 3 else None


class VpsQuery(SandboxStringParameter):
    name = 'vps_query'
    description = 'Vps query'
    default_value = 'vp_ids=AppSearchView&appwiz={{"force":1}}&app_platform={platform}&device={device}&vp_extra=MOBILE_SAAS_APPINFO%3AappId%3D{id}%7Ccountry%3D{country}'


class SnippetDisabled(SandboxBoolParameter):
    name = 'snippet_disabled'
    description = 'SerpInfo disabled'
    default_value = True


class Platform(SandboxRadioParameter):
    name = 'device_platform'
    devices = {
        'ipad': ['tablet', 'iOS'],
        'apad': ['tablet', 'Android'],
        'iphone': ['touch', 'iOS'],
        'android': ['touch', 'Android']
    }
    choices = [(device, device) for device in devices.iterkeys()]
    description = 'Platform'


class DataInConstruct(SandboxBoolParameter):
    name = 'data_in_construct'
    description = 'Place data in construct'
    default_value = False


class BnoAppBuildTask(BnoBaseBuildTask):
    type = 'BNO_APP_BUILD'
    input_parameters = BnoBaseBuildTask.input_parameters + [VpsQuery, Platform, SnippetDisabled, DataInConstruct]

    snippet_disabled = True
    data_in_construct = False
    keysemantic = 'snipdocid'
    namespace = 'app_bno'
    content_plugin = {
        'content_plugin': True,
        'disabled': True,
        'allowed_positions': [0],
        'filter': None,
        'SerpInfo': {
            'type': 'construct',
            'format': 'json',
            'template': 'app_bno_view',
            'adapter': 'app_bno_view'
        },
        'SerpData': {
            'type': 'app_bno_view',
            'viewport': None
        }
    }

    touch_info = {
        'type': 'bno',
        'format': 'json',
        'slot': 'post',
        'slot_rank': 2,
        'flat': 1,
        'kind': 'snippets',
        'adapter': 'bno'
    }

    touch_json = {
        'description': None,
        'icon': None,
        'id': None,
        'name': None,
        'price': {
            'currency': None,
            'value': None
        },
        'rating': {
            'value': None,
            'votes': None
        }
    }

    filter = 'rearr.device == {} && rearr.dd_osfamily == {}'

    def on_execute(self):
        self.data_in_construct = self.ctx[DataInConstruct.name]
        suffix = '' if not self.data_in_construct else '_' + 'new'
        self.namespace = self.namespace + '_' + self.ctx[Platform.name] + suffix
        self.snippet_disabled = self.ctx[SnippetDisabled.name]
        BnoBaseBuildTask.on_execute(self)

    def build_queries(self):
        # <docid><tab><gplay><tab><store>
        resource = sync_resource(get_last_resource(BNO_URL_TO_APP, {'arch': None}).id)
        with codecs.open(resource, 'r', encoding="utf-8") as f:
            data = [item[:-1].split('\t') for item in f]
        data = [[item[0], parse_gplay(item[1]), parse_istore(item[2]), parse_istore(item[3])] for item in data]

        # docid {gplay} {store} platform device
        data = [item + [self.ctx[Platform.name], Platform.devices[self.ctx[Platform.name]][0]] for item in data]

        # docid {gplay}|{store} platform device
        data = [[item[0], item[1], item[4], item[5]] for item in data if item[1] and item[4] in 'android apad'] + \
               [[item[0], item[2], item[4], item[5]] for item in data if item[2] and item[4] in 'iphone'] + \
               [[item[0], item[3], item[4], item[5]] for item in data if item[3] and item[4] in 'ipad']

        # docid id country platform device
        data = [[item[0], item[1]['id'], item[1]['country'], item[2], item[3]] for item in data]

        # docid id country platform device url
        data = [
            (item + [self.build_url(id=item[1], country=item[2], platform=item[3], device=item[4]), ])
            for item in data]
        return data, save_resource(self, data, 'queries.txt')

    def build_url(self, **kwargs):
        query = self.ctx[VpsQuery.name].format(**kwargs)
        return 'http://{}/vps?{}'.format(self.ctx[VpsHost.name], query)

    def prepare_data(self, data):
        # docid data
        data = ([item[0], self.build_value(item)] for item in data if item[-1])
        data = [item for item in data if item[-1]]
        return data, data and save_resource(self, data, self.namespace + '.txt') or None

    @staticmethod
    def extract_appid(url, store):
        split = urlparse.urlsplit(url)
        if store == 'gplay':
            params = urlparse.parse_qs(split.query)
            return params['id'][0] if 'id' in params else None

        path = os.path.split(split.path)[1]
        return path[2:] if path.startswith('id') else path

    def reformat_json(self, json, store):
        result = self.touch_json.copy()
        badge = json['appbadgeCard']['badge']
        result['icon'] = badge['icon']['src']
        result['name'] = badge['title']['text']
        result['id'] = self.extract_appid(badge['@actions']['buy']['data'], store)
        if 'description' in badge:
            result['description'] = badge['description']['text']

        if 'price' in badge:
            result['price']['currency'] = badge['price']['unit']
            result['price']['value'] = badge['price']['value']

        if 'rating' in badge:
            rating = badge['rating']
            if 'value' in rating:
                result['rating']['value'] = rating['value']
            if 'votes' in rating:
                result['rating']['votes'] = rating['votes']
        return result

    @staticmethod
    def hl_text(data, key):
        if key not in data:
            return

        data[key] = {'__hl': data[key]}

    def hl_viewport(self, json):
        badge = json['appbadgeCard']['badge']
        self.hl_text(badge['title'], 'text')
        if 'description' in badge:
            self.hl_text(badge['description'], 'text')

    def build_value(self, item):
        viewport = json.loads(item[-1]).get('AppSearchView')
        if not viewport:
            return None
        plugin = self.content_plugin.copy()
        plugin['disabled'] = self.snippet_disabled
        platform = Platform.devices[item[3]][-1]
        plugin['filter'] = self.filter.format(item[4], platform)
        if item[3] == 'apad' or item[3] == 'ipad':
            self.hl_viewport(viewport)
            plugin['SerpData']['viewport'] = viewport
        else:
            store = 'gplay' if platform == 'Android' else 'itunes'
            if self.data_in_construct:
                plugin['SerpData'].pop('viewport', None)
                plugin['SerpData']['type'] = 'bno'
                plugin['SerpData']['mobile_apps'] = {store: self.reformat_json(viewport, store)}
            else:
                plugin['SerpData'] = {'mobile_apps': {store: self.reformat_json(viewport, store)}}
                plugin['SerpInfo'] = self.touch_info.copy()
        return 'Snippet={"%s": %s}' % (self.namespace, json.dumps(plugin, ensure_ascii=False))


__Task__ = BnoAppBuildTask
