import codecs
import json
import urllib
import urlparse

from sandbox.projects.common.apihelpers import get_last_resource
from sandbox.projects.common.utils import sync_resource

from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter

from sandbox.projects.common.bno.task import BnoBaseBuildTask
from sandbox.projects.common.bno.resources import save_resource
from sandbox.projects.resource_types import BNO_NEWS_HOSTS


class VpsHost(SandboxStringParameter):
    name = 'vps_host'
    description = 'Vps host'
    default_value = 'vps-beta.n.yandex-team.ru'


class VpsQuery(SandboxStringParameter):
    name = 'vps_query'
    description = 'Vps query'
    default_value = 'vp_ids=NewsBnoDirectView&vp_extra=ONTO_HOST_NEWS_SEARCH%3Ahost%3D{host}{federal}'


class IsDisabled(SandboxBoolParameter):
    name = 'is_disabled'
    description = 'Is content plugin disabled'
    default_value = True


class AdditionalData(SandboxStringParameter):
    name = 'add_data'
    description = 'Additional data'


class BnoNewsBuildTask(BnoBaseBuildTask):
    input_parameters = BnoBaseBuildTask.input_parameters + [VpsQuery, IsDisabled, AdditionalData]
    type = 'BNO_NEWS_BUILD'
    namespace = 'news_bno'
    content_plugin = {
        namespace: {
            'content_plugin': True,
            'disabled': True,
            'allowed_positions': [0],
            'SerpInfo': {
                'type': 'construct',
                'format': 'json',
                'template': 'news_bno_view',
                'adapter': 'news_bno_view'
            },
            'SerpData': {
                'type': 'news_bno_view',
                'viewport': None
            }
        }
    }

    def build_url(self, **kwargs):
        query = self.ctx[VpsQuery.name].format(**kwargs)
        return 'http://{}/vps?{}'.format(self.ctx[VpsHost.name], query)

    def build_queries(self):
        resource = sync_resource(get_last_resource(BNO_NEWS_HOSTS, {'arch': None}).id)
        with codecs.open(resource, 'r', encoding="utf-8") as f:
            data = (item[:-1].split('\t') for item in f)
            urls = list(data)
        urls = [url + [urlparse.urlparse(url[1]).netloc] for url in urls]
        urls = [url + [
            self.build_url(host=urllib.quote(url[-1].encode("utf-8")), federal='%7Cfed%3Dda' if url[3] == '1' else '')]
                for url in urls]
        return urls, save_resource(self, urls, 'queries.txt')

    def build_value(self, value):
        viewport = json.loads(value).get('NewsBnoDirectView')
        if not viewport:
            return None
        plugin = self.content_plugin.copy()
        plugin[self.namespace]['disabled'] = self.ctx[IsDisabled.name]
        plugin[self.namespace]['SerpData']['viewport'] = viewport
        return 'Snippet=' + json.dumps(plugin, ensure_ascii=False)

    def prepare_data(self, data):
        data = [(item[2], self.build_value(item[-1])) for item in data]
        for resource_type in self.ctx[AdditionalData.name].split():
            data += self.get_prepared_data(resource_type)
        return data, data and save_resource(self, data, self.namespace + '.txt') or None

    @staticmethod
    def get_prepared_data(resource_type):
        resource = sync_resource(get_last_resource(resource_type, {'arch': None}).id)
        with codecs.open(resource, 'r', encoding="utf-8") as f:
            return [item[:-1].split('\t') for item in f]


__Task__ = BnoNewsBuildTask
