# -*- coding: utf-8 -*-

from sandbox import sdk2
import sandbox.common.types.resource as ctr
import sandbox.projects.resource_types as rt

from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.yabs.app_options import AppOptions

import json
import tarfile


class BroadmatchOverdraftBin(rt.ARCADIA_PROJECT):
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = ["kokokosttn", "olegts", "serkh"]
    release_subscribers = ["kokokosttn"]


class BroadmatchOverdraft(sdk2.Task):
    class Requirements(sdk2.Requirements):
        pass

    class Parameters(sdk2.Parameters):
        binary_id = sdk2.parameters.LastReleasedResource(
            'broadmatch overdraft dumping binary',
            resource_type=BroadmatchOverdraftBin,
            state=(ctr.State.READY, ),
            required=True
        )
        period = sdk2.parameters.Integer(
            "period (The overdraft is calculated using statistics collected for 'period' hours from ('now' - 'offset') hours)",
            default=1,
        )
        offset = sdk2.parameters.Integer(
            'offset',
            default=2,
        )
        suffix = sdk2.parameters.String(
            'This word will be appended to all the output tables/graphs names',
            default='Hourly',
        )
        ttl = sdk2.parameters.Integer(
            'TTL for BroadmatchOverdraft%%broadmatch_overdraft_suffix%% yt table in days',
            default=90,
        )
        cluster = sdk2.parameters.String(
            'Cluster the script will work with',
            default='alyx',
        )
        graphite_frequency = sdk2.parameters.String(
            'Graphite frequency: https://wiki.yandex-team.ru/dljaadminov/paysys/graphite/#konvencijaimenovanija',
            default='one_hour',
        )

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.binary_id)

        env = {'YT_TOKEN': sdk2.Vault.data('robot-yabs-yt-rw-token')}

        app_options = sdk2.ResourceData(sdk2.Resource.find(type=AppOptions.RESOURCE_TYPE, state=ctr.State.READY).first())

        with tarfile.open(str(app_options.path), 'r:gz') as tar:
            tar.extractall()

        app_options_file = AppOptions.make_app_options_file("sandbox")

        with open(app_options_file, "r") as file:
            host_options = json.load(file)
            alpha = host_options.get("broadmatch-synonym-alpha", 0)

        cmd = [
                str(bin_res.path),
                '--graphite_frequency={}'.format(self.Parameters.graphite_frequency),
                '--cluster={}'.format(self.Parameters.cluster),
                '--ttl={}'.format(self.Parameters.ttl),
                '--suffix={}'.format(self.Parameters.suffix),
                '--offset={}'.format(self.Parameters.offset),
                '--period={}'.format(self.Parameters.period),
                '--alpha={}'.format(alpha),
        ]

        with sdk2.helpers.ProcessLog(self, logger="broadmatch_overdraft") as l:
            sp.check_call(cmd, stdout=l.stdout, stderr=l.stderr, env=env)
