#!/usr/bin/env python
# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

TVM_SRC_ID = 2036368
# webmaster production tvm id
TVM_DST_ID = 2000036


def get_service_ticket(src_tvm_id, src_tvm_secret, dst_tvm_id):
    from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
    import os
    import subprocess
    hpe = HermeticPythonEnvironment(python_version='2.7.17', packages=('tvmauth==3.2.3',))
    with hpe:
        script_path = os.path.join(os.path.dirname(__file__), 'get_tvm_ticket.py')
        return subprocess.check_output([
            str(hpe.python_executable), script_path,
            str(src_tvm_id), src_tvm_secret, str(dst_tvm_id)]).strip()


class CertLogAlert(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):

        environments = (PipEnvironment("Jinja2"),
                        PipEnvironment("yandex-yt"),
                        PipEnvironment("yandex-yt-yson-bindings-skynet"),
                        PipEnvironment("sandbox-library"),
                        # PipEnvironment("cryptography"),
                        # check https://st.yandex-team.ru/DEVTOOLSSUPPORT-5219#5fd35214aa62c1459c71c34b for explanation
                        PipEnvironment("startrek_client", '2.5',
                                       custom_parameters=['--upgrade-strategy only-if-needed']))

    class Parameters(sdk2.Task.Parameters):
        st_api_url = sdk2.parameters.String("STARTREK base api url", required=False,
                                            default_value="https://st-api.yandex-team.ru/")
        yt_cluster_name = sdk2.parameters.String("YT cluster", required=True, default_value="hahn")
        yt_table_name = sdk2.parameters.String("YT table to parse", required=False,
                                               default_value="//home/infrasec/naumov-al/cert_chain_test")
        is_debug = sdk2.parameters.Bool("Debug", required=False, default_value=False)
        src_tvm_id = sdk2.parameters.Integer("TVM SRC ID", required=True, default_value=TVM_SRC_ID)
        dst_tvm_id = sdk2.parameters.Integer("TVM DST ID", required=True, default_value=TVM_DST_ID)

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.owner, "YtOauthToken")
        st_token = sdk2.Vault.data(self.owner, "StOauthToken")
        tvm_secret = sdk2.Vault.data(self.owner, "TvmClientId")
        import yt.wrapper as yt
        from checker import Checker
        from entry import Entry
        from alert import Alert
        client = yt.YtClient(proxy=self.Parameters.yt_cluster_name, token=yt_token)
        path = yt.TablePath(self.Parameters.yt_table_name)
        rows = client.read_table(path)
        service_ticket = get_service_ticket(self.Parameters.src_tvm_id, tvm_secret, self.Parameters.dst_tvm_id)
        checker = Checker({}, service_ticket)
        checker.apply_config()

        for row in rows:
            entry = Entry(row)
            checker.check(entry)

        detects = checker.get_detects()
        print("{} detects found".format(len(detects)))
        if len(detects) == 0:
            return

        alert = Alert(st_token, self.Parameters.st_api_url)
        alert.proceed_detects(detects)
