#!/usr/bin/env python
# -*- coding: utf-8 -*-
from cryptography import x509
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes
from cryptography.x509.oid import ExtensionOID

EXTENSIONS_MAP = {
    # ExtensionOID.SIGNED_CERTIFICATE_TIMESTAMPS: "SCT_EXT",
    ExtensionOID.PRECERT_SIGNED_CERTIFICATE_TIMESTAMPS: "PreSCT_EXT"
}


class CertificateChain(object):
    def __init__(self, chain):
        self.chain = []
        self.sct_list = []
        for cert in chain:
            c = x509.load_pem_x509_certificate(cert, default_backend())
            self.chain.append(c)

        if len(self.chain) > 0:
            self.leaf = self.chain[0]
            for ext_name in EXTENSIONS_MAP:
                try:
                    ext = self.leaf.extensions.get_extension_for_oid(ext_name)
                    # ext.value = list of SignedCertificateTimestamp objects
                    self.sct_list = ext.value
                except Exception as ex:
                    print("Unable to extract SCT extension {exc} for {domain}".format(exc=ex, domain=self.leaf.subject))

    def get_leaf(self):
        return self.leaf

    def get_chain(self):
        return self.chain

    def get_sct(self):
        return self.sct_list

    def get_leaf_fingerprint(self):
        return self.leaf.fingerprint(hashes.SHA256())
