#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import re
import requests


class CommitSearch(object):
    src_url = "https://chromium.googlesource.com"
    cr_log_path = "/chromium/src/+log/"
    cr_commit_path = "/chromium/src/+/"
    v8_log_path = "/v8/v8/+log/"
    v8_commit_path = "/v8/v8/+/"
    pdfium_src_url = "https://pdfium.googlesource.com"
    pdfium_log_path = "/pdfium/+log/"
    pdfium_commit_path = "/pdfium/+/"
    id_re = re.compile(r'\d+')

    def __init__(self, product, version, bug, max_pages=250):
        self.page_counter = max_pages
        self.found = False
        self.commits = set()
        self.id = bug
        self.bug_sbustrs = {"bug:", "bug="}
        self.changelog_lookup_depth = 128
        self.requests_timeout = 5
        if product == "chromium":
            self.rel_url = self.src_url + self.cr_log_path + version
            self.commit_url = self.src_url + self.cr_commit_path

        if product == "v8":
            self.rel_url = self.src_url + self.v8_log_path + version
            self.commit_url = self.src_url + self.v8_commit_path

        if product == "pdfium":
            self.rel_url = self.pdfium_src_url + self.pdfium_log_path + version
            self.commit_url = self.pdfium_src_url + self.pdfium_commit_path

    def _extract_bug_string(self, message):
        for item in self.bug_sbustrs:
            bug_pos = message.lower().find(item)
            if bug_pos <= 0:
                continue

            bug_part = message[bug_pos:]
            bug_string = bug_part.split("\n")[0]
            return bug_string

        return None

    def _find_bug_id(self, commits):
        for commit in commits:
            message = commit.get("message")
            if not message:
                continue

            bug_str = self._extract_bug_string(message)
            if not bug_str:
                continue

            matched = re.findall(self.id_re, bug_str)
            if not matched:
                continue

            for item in matched:
                if item == self.id:
                    sha1 = commit.get("commit")
                    self.commits.add(self.commit_url + sha1)
                    self.found = True

    def _parse_changelog(self, log, depth):
        if self.found or depth < 0:
            return

        self.page_counter -= 1
        next_id = log.get("next")
        commits = log.get("log")
        if commits:
            self._find_bug_id(commits)

        if next_id:
            next_changelog_url = self.rel_url + "?s=" + next_id + "&format=JSON"
            try:
                next_log = json.loads(requests.get(next_changelog_url, timeout=self.requests_timeout).content[4:])

            except Exception as ex:
                print("Load changelog from url {} failed: {}".format(next_changelog_url, ex))
                return

            if self.page_counter > 0:
                try:
                    self._parse_changelog(next_log, depth-1)

                except RuntimeError:
                    return

    def find_bug_commit(self):
        changelog_url = self.rel_url + "/?format=JSON"
        try:
            log = json.loads(requests.get(changelog_url, timeout=self.requests_timeout).content[4:])
        except Exception as ex:
            print("Exception raised while finding changelog: {}".format(ex))
            return

        self._parse_changelog(log, self.changelog_lookup_depth)
        return self.commits
