# -*- coding: utf-8 -*-

from os import path
import logging
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.environments import VirtualEnvironment
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process

ARCHIVE = 'server.tar.gz'
TAR_MODE = 'w:gz'


class GitBranch(SandboxStringParameter):
    name = "branch"
    description = "branch"
    default_value = "master"

# Sandbox task


class BuildAbgameBackend(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_ABGAME_BACKEND'
    input_parameters = [GitBranch]
    execution_space = 1200

    GIT_URL = 'https://github.yandex-team.ru/Rock/abgame_backend.git'
    _checkout_path = 'bin/server/'

    @property
    def sources(self):
        return self._checkout_path

    def _create_tar(self, filenames, tarname, tarmode=TAR_MODE):
        tar = tarfile.open(tarname, tarmode)
        for fname in filenames:
            logging.info('Added %s to archive', fname)
            tar.add(fname)
        tar.close()

    def _checkout_sources(self, tag):
        logging.info('Checking out %s tag of the repository', tag)

        src_dir = self.path('backend')
        copy_dir = self.path('backend/src')

        run_process(['git', 'clone', self.GIT_URL, src_dir], log_prefix='git_clone')
        run_process(['git', 'checkout', tag], work_dir=src_dir, log_prefix='git_checkout')

        copy_path(copy_dir, self.sources)

    def on_execute(self):
        self._checkout_sources(self.ctx['branch'])

        with VirtualEnvironment(do_not_remove=True) as venv:
            venv.pip('-U pip')
            venv.pip('-r {}'.format(path.join(self.sources, "requirements.txt")))

            files = [path.basename(venv.root_dir), self.sources]
            self._create_tar(files, tarname=ARCHIVE)

        self.create_resource(
            description='ABGAME_BACKEND_PACKAGE',
            resource_path=ARCHIVE,
            resource_type=resource_types.ABGAME_BACKEND_PACKAGE,
            arch='linux'
        )


__Task__ = BuildAbgameBackend
