# -*- coding: utf-8 -*-

import os
import logging
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process

ARCHIVE = 'frontend.tar.gz'
ARCHIVE_TESTING = 'frontend-testing.tar.gz'
TAR_MODE = 'w:gz'


def unarchive(fname, newfname=None):
    if tarfile.is_tarfile(fname):
        if not newfname:
            idx = fname.find('.tar')
            newfname = fname[:idx]
        tar = tarfile.open(fname)
        tar.extractall(path=newfname)
        tar.close()
        os.unlink(fname)
        return newfname
    return fname

# Sandbox parameters


class NodeJSPackage(LastReleasedResource):
    name = 'YASAP_PDB_NODEJS_PACKAGE'
    description = 'node.js'
    resource_type = resource_types.YASAP_PDB_NODEJS_PACKAGE
    limit = 5


class GitBranch(SandboxStringParameter):
    name = "branch"
    description = "branch"
    default_value = "master"


class ShouldRebase(SandboxBoolParameter):
    name = "rebase"
    description = "rebase"
    default_value = False


class ShouldIgnoreRebaseErrors(SandboxBoolParameter):
    name = "rebase_ignore"
    description = "ignore rebase errors"
    default_value = False

# Sandbox task


class BuildAbgameFrontend(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_ABGAME_FRONTEND'
    GIT_URL = 'https://github.yandex-team.ru/Rock/abgame-frontend.git'

    input_parameters = [NodeJSPackage, GitBranch, ShouldRebase, ShouldIgnoreRebaseErrors]

    execution_space = 1200

    def _create_tar(self, filenames, tarname, tarmode=TAR_MODE):
        tar = tarfile.open(tarname, tarmode)
        for fname in filenames:
            logging.info('Added %s to archive', fname)
            tar.add(fname)
        tar.close()

    def _sync_res(self, resource):
        path = self.sync_resource(self.ctx[resource.name])
        copy_path(path, self.path())
        return path

    def _checkout_sources(self, tag, rebase, rebase_ignore):
        logging.info('Checking out %s tag of the repository', tag)
        src_dir = self.path('frontend')

        run_process(['git', 'clone', self.GIT_URL, src_dir], log_prefix='git_clone')
        run_process(['git', 'checkout', tag], work_dir=src_dir, log_prefix='git_checkout')

        try:
            if rebase:
                run_process(['git', 'config', '--global', 'user.email', 'polubob@yandex.ru'], work_dir=src_dir, log_prefix='git_config')
                run_process(['git', 'config', '--global', 'user.name', 'polubob'], work_dir=src_dir, log_prefix='git_config')
                run_process(['git', 'rebase', 'master'], work_dir=src_dir, log_prefix='git_rebase')
        except SandboxSubprocessError:
            if rebase_ignore:
                run_process(['git', 'rebase', '--abort'], work_dir=src_dir, log_prefix='git_rebase_abort')
            else:
                raise

        return src_dir

    def on_execute(self):
        files = []

        nodeArchive = self._sync_res(NodeJSPackage)
        nodeBaseName = os.path.basename(nodeArchive)
        nodeRoot = unarchive(nodeBaseName)
        node = os.path.abspath(os.path.join(nodeRoot, 'bin/node'))
        npm = os.path.abspath(os.path.join(nodeRoot, 'bin/npm'))
        binRoot = os.path.abspath(os.path.join(nodeRoot, 'bin'))

        sources_folder = self._checkout_sources(self.ctx['branch'], self.ctx['rebase'], self.ctx['rebase_ignore'])

        environment = os.environ.copy()
        environment['PATH'] = binRoot + ':' + os.environ['PATH']

        files.append(os.path.basename(nodeRoot))
        files.append(os.path.basename(sources_folder))

        # Install dependencies
        run_process(
            [node, npm, '--registry=https://npm.yandex-team.ru', 'install', '--production'],
            log_prefix='npm_install',
            work_dir=sources_folder,
            environment=environment
        )

        # Build production
        run_process(
            [node, npm, 'run', 'release'],
            log_prefix='release',
            work_dir=sources_folder,
            environment=environment
        )

        # Build archive
        self._create_tar(files, tarname=ARCHIVE)

        # Create resource
        self.create_resource(
            description='ABGAME_FRONTEND_PACKAGE',
            resource_path=ARCHIVE,
            resource_type=resource_types.ABGAME_FRONTEND_PACKAGE,
            arch='any'
        )

        # Build testing
        run_process(
            [node, npm, 'run', 'release:testing'],
            log_prefix='release',
            work_dir=sources_folder,
            environment=environment
        )

        # Build archive
        self._create_tar(files, tarname=ARCHIVE_TESTING)

        # Create resource
        self.create_resource(
            description='ABGAME_FRONTEND_TESTING_PACKAGE',
            resource_path=ARCHIVE_TESTING,
            resource_type=resource_types.ABGAME_FRONTEND_TESTING_PACKAGE,
            arch='any'
        )

        pass


__Task__ = BuildAbgameFrontend
