# coding: utf-8
import os.path
import tarfile

from sandbox import sdk2
from sandbox.projects import resource_types as rt
from sandbox.projects.geobase.Geodata4BinStable.resource import GEODATA4BIN_STABLE
from sandbox.projects.geosearch import resource_types as grt


class BuildAddrSnippetDataRequester(sdk2.Task):
    GEODATA_BIN_PATH = 'geodata4.bin'

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4 * 1024  # 4 GB
        disk_space = 35 * 1024  # 35 GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 0
        notifications = []

        input_resource = sdk2.parameters.Resource('Resource with input data', resource_type=rt.OTHER_RESOURCE, required=True)
        metasearch_rps = sdk2.parameters.Integer('Maximum RPS to metasearch (makes sense in geojson mode only)', default=50, required=True)
        use_full_export = sdk2.parameters.Bool('Use full sprav export', default=True)
        debug_mode = sdk2.parameters.Bool('Enable debug logging', default=False)
        make_addrsnip_data_executable = sdk2.parameters.LastReleasedResource('make_addrsnip_data executable', resource_type=grt.MAKE_ADDRSNIP_DATA_EXECUTABLE, required=True)
        geobase = sdk2.parameters.LastReleasedResource('Geobase', resource_type=GEODATA4BIN_STABLE, required=True)

    def extract_geobase(self):
        geobase = sdk2.ResourceData(self.Parameters.geobase)
        with tarfile.open(str(geobase.path), 'r') as tar:
            tar.extractall()
        assert os.path.isfile(self.GEODATA_BIN_PATH)

    def on_execute(self):
        self.extract_geobase()
        infile = sdk2.ResourceData(self.Parameters.input_resource)
        make_addrsnip_data_exe = sdk2.ResourceData(self.Parameters.make_addrsnip_data_executable)

        params = [
            str(make_addrsnip_data_exe.path),
            '--stage', '2',
            '--infile', str(infile.path),
            '--geodata', self.GEODATA_BIN_PATH,
            '--querydata_key', 'addrsnip',
            '--main_url_full_export_querydata_key', 'addrsnip_main_url_full_export',
            '--all_urls_old_export_querydata_key', 'addrsnip_all_urls_old_export',
            '--all_urls_full_export_querydata_key', 'addrsnip_all_urls_full_export',
            '--max_rps', str(self.Parameters.metasearch_rps),
        ]
        if self.Parameters.use_full_export:
            params.append('--full-export')
        if self.Parameters.debug_mode:
            params.append('--debug')

        resource = rt.PLAIN_TEXT(self, 'raw json (part)', 'raw.json')
        with resource.path.open('w') as fout, self.log_path('stderr.log').open('w') as ferr:
            sdk2.helpers.subprocess.check_call(params, stdout=fout, stderr=ferr)
