# coding: utf-8
import os
import json
import logging
import tarfile

from sandbox.common.types.client import Tag
from sandbox.yasandbox import manager
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.projects import resource_types
from sandbox.projects.common import environments
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny import client
from sandbox.projects.common.nanny import const
from sandbox.projects.common.nanny.nanny import StartrekTicketIdsParameter
from sandbox.projects.TestAlemate import AlemateTestRunner


class GitRefIdParameter(parameters.SandboxStringParameter):
    name = 'ref_id'
    description = 'Git ref id'
    default_value = 'master'
    required = True


class GitRefShaParameter(parameters.SandboxStringParameter):
    name = 'ref_sha'
    description = 'Git ref SHA'
    default_value = ''
    required = False


class ReleaseParameter(parameters.SandboxBoolParameter):
    name = 'release'
    description = 'Create resource after tests'
    default_value = False


class BuildAlemate(nanny.ReleaseToNannyTask, SandboxTask):
    """
    Задача сборки кубиков.

    При сборке создаётся пустой virtualenv и ставится alemate

    При этом ставятся и все зависимости (alemate_api, sepelib, etc.)
    """
    type = 'BUILD_ALEMATE'

    input_parameters = [
        GitRefIdParameter,
        GitRefShaParameter,
        StartrekTicketIdsParameter,
        ReleaseParameter,
    ]

    execution_space = 4096

    environment = [
        environments.SandboxJavaJdkEnvironment('1.8.0'),
        environments.SwatMongoDbEnvironment('3.4.2'),
        environments.SwatZookeeperEnvironment('3.4.6'),
    ]

    release_to = ['alonger']

    TGZ_PATH = 'alemate.tar.gz'

    client_tags = Tag.LINUX_XENIAL
    cores = 1

    def on_execute(self):
        """
        Plan is:

        * git clone and checkout specified tag
        * build virtualenv
        * run unittests
        * build resource
        """
        sources_path = self.path('alemate')
        virtualenv_path = self.path('venv')
        test_runner = AlemateTestRunner()

        logging.info('Checking out the source code')
        oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
        test_runner.checkout(oauth_token=oauth_token,
                             dest=sources_path,
                             ref=self.ctx.get('ref_sha') or self.ctx['ref_id'])

        logging.info('Building virtualenv')
        test_runner.build_virtualenv(sources_path=sources_path,
                                     virtualenv_path=virtualenv_path)

        logging.info('Running unit tests')
        test_runner.run_tests(task_context=self.ctx,
                              virtualenv_path=virtualenv_path,
                              sources_path=sources_path,
                              test_script='test.sh',
                              timeout=600)

        if self.ctx.get('release'):
            run_process(['/skynet/python/bin/virtualenv', '--relocatable', virtualenv_path])
            ref_id = self.ctx['ref_id']
            if ref_id.startswith('refs/tags/'):
                tag = ref_id[len('refs/tags/'):]
            else:
                tag = 'master'
            description = 'alemate virtualenv tgz {0}'.format(tag)
            self._make_resource(virtualenv_path, description)

    def _make_resource(self, virtualenv_path, description):
        logging.info('creating tgz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
        self.create_resource(
            description=description,
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.ALEMATE,
            arch='linux',
        )

    def arcadia_info(self):
        """
        Получение информации о задаче при релизе

        :return список из трёх значений revision, tag, branch
        """
        return None, self.ctx.get('tag'), None

    def on_release(self, additional_parameters):
        logging.info('Gathering information about release of task %s to send it to Nanny', self.id)
        data = self.get_nanny_release_info(additional_parameters)
        logging.info('Release payload: %s', json.dumps(data, indent=4))
        logging.info('Sending release of task %s to production nanny', self.id)
        self.nanny_client.create_release2(data)
        logging.info('Release of task %s has been sent to production nanny', self.id)

        logging.info('Sending release of task %s to dev-nanny', self.id)
        dev_nanny_oauth_token = self.get_vault_data('GBG', 'nanny_robot_dev_nanny_oauth_token')
        dev_nanny_client = client.NannyClient(api_url=const.NANNY_DEV_API_URL, oauth_token=dev_nanny_oauth_token)
        dev_nanny_client.create_release2(data)
        logging.info('Release of task %s has been sent to dev-nanny', self.id)

        logging.info('Sending release of task %s to adm-nanny', self.id)
        adm_nanny_oauth_token = self.get_vault_data('GBG', 'nanny_robot_adm_nanny_oauth_token')
        adm_nanny_client = client.NannyClient(api_url='http://adm-nanny.yandex-team.ru/',
                                              oauth_token=adm_nanny_oauth_token)
        adm_nanny_client.create_release2(data)
        logging.info('Release of task %s has been sent to adm-nanny', self.id)

        self.mark_released_resources(additional_parameters["release_status"])
        self.set_logs_inf_ttl()

    def set_logs_inf_ttl(self):
        resources = [_ for _ in self.list_resources()]
        for resource in resources:
            if resource.type != 'TASK_LOGS':
                continue
            resource.attrs["ttl"] = 'inf'
            resource.attrs.setdefault("backup_task", True)
            manager.resource_manager.update(resource)
        return resources


__Task__ = BuildAlemate
