# -*- coding: utf-8 -*-

import logging
import re
import os.path

from sandbox.projects import resource_types

from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"


class BuildAnalyzethisIPReg(SandboxTask):
    '''
    Build IPREG like file for Ashmanov's ips
    '''

    type = 'ANALYZETHIS_IPREG'

    input_parameters = [ArcadiaUrlParameter, ]

    URL_PRIVILEGED_IPS_SVN = 'antirobot/scripts/whitelist_mr/data'
    PRIVILEGED_IPS_FILE = 'privileged_ips'
    IPREG_ANALYZETHIS_FILE = 'IPREG.analyzethis'

    ANALYZETHIS_REG_ID = 4294967295

    def _parse_privileged_ips_file(self, fpath):
        IPV6_IPV4_PREFIX = '::ffff:'

        def parse_ipv4_format(s):
            a = s.split('.')
            if len(a) != 4:
                return None
            for x_n, x in enumerate(a):
                if not x.isdigit():
                    return None
                i = int(x)
                if i < 0 or i > 255:
                    return None
                a[x_n] = hex(i)[2:]
            return IPV6_IPV4_PREFIX + '{}{}:{}{}'.format(*a)

        def ipreg_sort(a, b):
            a = int(''.join([x.rjust(4, '0') for x in a[len(IPV6_IPV4_PREFIX):a.find(' ')].split(':')]), 16)
            b = int(''.join([x.rjust(4, '0') for x in b[len(IPV6_IPV4_PREFIX):b.find(' ')].split(':')]), 16)
            return cmp(a, b)

        data = [s.rstrip('\r\n') for s in open(fpath)]
        if len(data) < 2:
            raise ValueError('privileged_ips file too small')
        if data[0] != '# Ashmanov':
            raise ValueError('privileged_ips file first line is not \'# Ashmanov\'')
        ret = []
        for row_num, row in enumerate(data[1:], 2):
            row = row.strip()
            if not row:
                break
            ips = [ip.strip() for ip in row.split('-')]
            if len(ips) not in (1, 2):
                raise ValueError('something wrong with privileged_ips file, row {}'.format(row_num))
            if len(ips) == 2:
                ip1, ip2 = ips
            else:
                ip1 = ips[0]
                ip2 = ips[0]
            ip1_ipv6 = parse_ipv4_format(ip1)
            if ip1_ipv6 is None:
                raise ValueError('wrong ipv4 format ({}) in privileged_ips file, row {}'.format(ip1, row_num))
            ip2_ipv6 = parse_ipv4_format(ip2)
            if ip2_ipv6 is None:
                raise ValueError('wrong ipv4 format ({}) in privileged_ips file, row {}'.format(ip2, row_num))
            ret.append('{} {} {}'.format(ip1_ipv6, ip2_ipv6, self.ANALYZETHIS_REG_ID))
        if not ret:
            raise ValueError('something wrong with privileged_ips file: no ips?')
        ret.sort(cmp=ipreg_sort)
        return '\n'.join(ret)

    def on_execute(self):
        logging.info('Build analyzethis ipreg')

        self._export_arcadia(os.path.join(self.URL_PRIVILEGED_IPS_SVN, self.PRIVILEGED_IPS_FILE),
                             self.PRIVILEGED_IPS_FILE)

        logging.info('Parsing privileged_ips_filepath')

        parsed = self._parse_privileged_ips_file(self.PRIVILEGED_IPS_FILE)

        analyzethis_ipreg_filepath = self.abs_path(self.IPREG_ANALYZETHIS_FILE)
        with open(analyzethis_ipreg_filepath, 'w') as f:
            f.write(parsed)

        logging.info('Creating resource')

        url_arcadia = self.ctx[SandboxArcadiaUrlParameter.name]
        revision = Arcadia.get_revision(url_arcadia)
        resource = self._create_resource('IPREG.analyzethis_{}'.format(revision),
                                         analyzethis_ipreg_filepath,
                                         resource_types.ANALYZETHIS_IPREG)
        self.ctx['config_resource_id'] = resource.id
        self.mark_resource_ready(resource)
        return resource.id

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[SandboxArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)


__Task__ = BuildAnalyzethisIPReg
