# -*- coding: utf-8 -*-

import json
import logging

from sandbox import common
from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import LastReleasedResource


class PrevAppHostBackendsConfig(LastReleasedResource):
    name = 'PREV_APP_HOST_BACKENDS_CONFIG'
    description = 'previous backends config'
    resource_type = resource_types.APP_HOST_BACKENDS_CONFIG
    limit = 5
    required = True


class NewAppHostBackendsConfig(parameters.ResourceSelector):
    name = 'NEW_APP_HOST_BACKENDS_CONFIG'
    description = 'new backends config (optional)'
    resource_type = resource_types.APP_HOST_BACKENDS_CONFIG
    limit = 5
    default_value = None
    required = False


class AppHostBackendsGeneratorConfig(LastReleasedResource):
    name = 'APP_HOST_BACKENDS_GENERATOR_CONFIG'
    description = 'config for backends generator'
    resource_type = resource_types.APP_HOST_BACKENDS_GENERATOR_CONFIG
    limit = 5
    required = True


class BuildAndDiffAppHostBackendsConfig(SandboxTask):
    type = "BUILD_AND_DIFF_APP_HOST_BACKENDS_CONFIG"
    execution_space = 1024

    input_parameters = [
        PrevAppHostBackendsConfig,
        AppHostBackendsGeneratorConfig,
        NewAppHostBackendsConfig
    ]

    def read_config(self, config_path):
        with open(config_path, 'r') as fh:
            return json.load(fh)

    def diff_instances(self, prev, new):
        p, n = {}, {}

        def get_backends_map(doc):
            ret = {}
            if 'meta_backend_descrs' in doc.keys():
                for group in doc[u'meta_backend_descrs']:
                    for i in group[u'backend_descrs']:
                        key = "{}:{}".format(str(i[u'host']), str(i[u'port']))
                        ret[key] = i
            else:
                for i in doc[u'backend_descrs']:
                    key = "{}:{}".format(str(i[u'host']), str(i[u'port']))
                    ret[key] = i
            return ret

        p = get_backends_map(prev)
        n = get_backends_map(new)

        added_instances = set(set(n.keys()) - set(p.keys()))
        removed_instances = set(set(p.keys()) - set(n.keys()))
        modified = []
        union_instances = set(p.keys()).intersection(set(n.keys()))
        for ikey in union_instances:
            diff = {}
            diff['prev'] = p.get(ikey)
            diff['new'] = n.get(ikey)
            diff['fields'] = []
            for field in n.get(ikey).keys():
                pf = p.get(ikey).get(field)
                nf = n.get(ikey).get(field)
                if pf != nf:
                    diff['fields'].append(field)
            if len(diff['fields']) != 0:
                modified.append(diff)
        added = [n.get(k) for k in added_instances]
        removed = [p.get(k) for k in removed_instances]
        return (added, removed, modified)

    def on_execute(self):
        prev = None
        new = None
        degrade_level = .1

        if self.ctx.get('new_config') is None \
                and self.ctx[NewAppHostBackendsConfig.name] is None:
            subtask = self.create_subtask(
                task_type='BUILD_APP_HOST_BACKENDS_CONFIG',
                description=self.descr,
                input_parameters=self.ctx
            )
            self.ctx['new_config'] = subtask.id
            logging.info(
                'Starting BUILD_APP_HOST_BACKENDS_CONFIG task {}'.format(
                    subtask))
            self.wait_task_completed(subtask)

        with self.current_action('Downloading new backends config'):
            resource_id = self.ctx[NewAppHostBackendsConfig.name]
            if self.ctx.get('new_config') is not None:
                sb = common.rest.Client()
                resources = \
                    sb.resource[
                        {
                            'type': 'APP_HOST_BACKENDS_CONFIG',
                            'task_id': self.ctx.get('new_config')
                        }, : 1]['items']
                logging.info(resources)
                resource_id = resources[0][u'id']
            path = self.sync_resource(resource_id)
            new = self.read_config(path)

        with self.current_action('Downloading previous backends config'):
            prev_config_path = \
                self.sync_resource(self.ctx[PrevAppHostBackendsConfig.name])
            prev = self.read_config(prev_config_path)

        with self.current_action('Comparing backends configs'):
            # Comparing configurations count and names
            auto_release = True
            changes = False
            reasons = []
            union = set(prev.keys()).intersection(set(new.keys()))
            added = set(set(new.keys()) - set(prev.keys()))
            removed = set(set(prev.keys()) - set(new.keys()))
            if len(added) != 0 or len(removed) != 0:
                auto_release = False
                changes = True
            if len(added) != 0:
                reasons.append('added configurations: {}'.format(added))
                logging.warn(reasons[-1])
            if len(removed) != 0:
                reasons.append('removed configurations: {}'.format(added))
                logging.warn(reasons[-1])
            if len(added) == 0 and len(removed) == 0:
                logging.info('configurations is the same')

            # Comparing sources count and names on union confs
            for conf in union:
                logging.info('comparing configuration `{}`'.format(conf))
                p = prev.get(conf)
                n = new.get(conf)
                union_sources = set(p.keys()).intersection(set(n.keys()))
                added = set(set(n.keys()) - set(p.keys()))
                removed = set(set(p.keys()) - set(n.keys()))
                if len(added) != 0 or len(removed) != 0:
                    auto_release = False
                    changes = True
                if len(added) != 0:
                    reasons.append('added sources: {}'.format(added))
                    logging.warn(reasons[-1])
                if len(removed) != 0:
                    reasons.append('removed sources: {}'.format(removed))
                    logging.warn(reasons[-1])

                logging.info('sources is the same')

                # check new sources are not bad
                for src in added:
                    n = new.get(conf).get(src)
                    if len(n) == 0:
                        auto_release = False
                        reasons.append(
                            'new source in config `{}` is empty'.format(src))
                        logging.warn(reasons[-1])
                # check union sources are not bad
                for src in union_sources:
                    p = prev.get(conf).get(src)
                    n = new.get(conf).get(src)
                    if len(n) == 0:
                        auto_release = False
                        reasons.append(
                            'source in new config `{}` is empty'.format(src))
                        logging.warn(reasons[-1])

                    abs_degrade_value = round(len(p) * degrade_level)
                    added, removed, modified = self.diff_instances(p, n)
                    logging.info(
                        'source `{}` added {} / removed {} / modified {}'
                        .format(src, len(added), len(removed), len(modified)))
                    if len(added) != 0 \
                            or len(removed) != 0 or len(modified) != 0:
                        changes = True
                    if len(added) > 0:
                        logging.info('+ added: {}'.format(
                            [x.get('host') for x in added])
                        )
                    if len(removed) >= abs_degrade_value:
                        auto_release = False
                        reasons.append(
                            'source `{}` config removed {} instances: {}'
                            .format(
                                src,
                                len(removed),
                                removed))
                        logging.warn(reasons[-1])
                    if len(removed) > 0:
                        logging.info('- removed {}'.format(
                            [x.get('host') for x in removed])
                        )
                    if len(modified) > 0:
                        for i in modified:
                            logging.info(
                                'modified instance {}:{}'
                                .format(i['prev']['host'], i['prev']['port']))
                            for f in i['fields']:
                                logging.info(
                                    '- {} : {}'
                                    .format(f, i['prev'].get(f)))
                                logging.info(
                                    '+ {} : {}'
                                    .format(f, i['new'].get(f)))
            logging.info(
                "auto_release={}, changes={}"
                .format(auto_release, changes))
            if auto_release and changes:
                logging.info('RELEASE IT!')
