# -*- coding: utf-8 -*-

import re
import os
import logging

import sandbox.common.types.client as ctc

from sandbox.projects.common.utils import svn_revision

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types

from sandbox.projects.common.nanny import nanny
import sandbox.projects.common.arcadia.sdk as arcadia_sdk


class DebugBundle(SandboxBoolParameter):
    name = 'debug_bundle'
    description = 'Create debug bundle'
    default_value = False


class BuildApacheBundle(nanny.ReleaseToNannyTask, SandboxTask):

    type = 'BUILD_APACHE_BUNDLE'

    input_parameters = (SandboxArcadiaUrlParameter, DebugBundle)
    execution_space = 70 * 1000  # 63Gb detected

    client_tags = ctc.Tag.LINUX_PRECISE

    def on_execute(self):
        arcadia_url = self.ctx[SandboxArcadiaUrlParameter.name]
        with arcadia_sdk.mount_arc_path(arcadia_url) as arcadia_src_dir:
            self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)

            work_dir = self.abs_path('workcopy')
            make_folder(work_dir)

            os.environ['SANDBOX_TASK_ID'] = str(self.id)
            if self.ctx.get(DebugBundle.name):
                os.environ['DEBUG_YA'] = '1'

            build_apache_cmd = [
                'bash',
                os.path.join(arcadia_src_dir, 'search', 'daemons', 'justapache', 'q_apache_bundle'),
                arcadia_src_dir,
                work_dir
            ]
            run_process(build_apache_cmd, log_prefix='run_q_apache_bundle')

            result_bundle_file = None
            for item in os.listdir(work_dir):
                file_path = os.path.join(work_dir, item)
                if os.path.isfile(file_path) and re.match('apache_bundle[.].*[.]tar[.]gz', item):
                    result_bundle_file = file_path
                    break
            if result_bundle_file:
                apache_bundle_resource = self.create_resource(
                    description=self.descr,
                    resource_path=result_bundle_file,
                    resource_type=resource_types.APACHE_BUNDLE,
                    arch=self.client_info['arch'],
                    attributes={
                        'version': self.get_apache_version(),
                    }
                )
            else:
                raise SandboxTaskFailureError('Unable to find apache bundle file "apache_bundle.*.tar.gz".')

            logging.info('Try to find dbg file')
            dbg_apache_file_path = os.path.join(work_dir, 'search', 'daemons', 'justapache', 'justapache.dbg')
            if os.path.exists(dbg_apache_file_path):
                description = 'apache debug file for task {} and apache resource {}'.format(
                    self.id, apache_bundle_resource.id
                )
                self.create_resource(
                    description=description,
                    resource_path=dbg_apache_file_path,
                    resource_type=resource_types.BUILD_APACHE_DEBUG_FILE,
                    arch=self.arch,
                )

    def get_apache_version(self):
        (rev, tag, branch) = self.arcadia_info()
        if branch is not None:
            return branch.split('/')[-1].split('-')[-1]
        if tag is not None:
            return tag.split('/')[-2].split('-')[-1]
        else:
            return 'trunk'

    def arcadia_info(self):
        parsed_url = Arcadia.parse_url(self.ctx[SandboxArcadiaUrlParameter.name])
        return parsed_url.revision, parsed_url.tag, parsed_url.branch

    def on_release(self, additional_parameters):
        if self.ctx.get(DebugBundle.name):
            raise SandboxTaskFailureError("Debug version couldn't be released.")

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = BuildApacheBundle
