import abc
import glob
import logging
import os.path
import requests

from sandbox.projects import resource_types
from sandbox.projects.common.releases import Releaser, ShouldMakeNewRelease
from sandbox.projects.common.app_host import gen_symdump, SymbolsDumper
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.parameters as build_params

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxSelectParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia, ArcadiaTestData
from sandbox.sdk2.resource import AbstractResource
from sandbox.projects.common.utils import check_if_tasks_are_ok

from atombuildconfig import process

_PERS_RERANKD = "pers-rerankd"
_ATOM_FRONT = "atom-front"

_DAEMON_NAMES = [
    _PERS_RERANKD,
    _ATOM_FRONT,
    "config",
    "hook",
    "prod-loop",
    "data-updater",
    "rerankd-pushclient-conf"
]

_DAEMON_COMPONENTS = {
    "all": _DAEMON_NAMES,
    _PERS_RERANKD: [_PERS_RERANKD, "prod-loop", "rerankd-pushclient-conf", "data-updater"],
    _ATOM_FRONT: [_ATOM_FRONT, "config", "hook"],
    "rerank-and-front": [_PERS_RERANKD, _ATOM_FRONT, "config", "hook", "prod-loop", "rerankd-pushclient-conf", "data-updater"],
    "config-and-hook": ["config", "hook", "data-updater", "prod-loop", "rerankd-pushclient-conf"]
}


class PERS_RERANK_SERVICE_PUSHCLIENT_CONF(AbstractResource):
    """
        Push-client config for rerankd instances
    """
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = resource_types.personal_releasers


class SelectDaemon(SandboxSelectParameter):

    required = True
    name = "daemon"
    description = "Daemon:"

    choices = [
        (x, x) for x in [
            "all", "rerank-and-front",
            "config-and-hook"
        ] + _DAEMON_NAMES
    ]
    default_value = choices[0][1]


class SymbolsOptionalDumper(SymbolsDumper):
    description = "Breakpad symbols dumper:"
    required = False


def copy_event_log_dump(build_dir, package_dir):
    copy_path(
        os.path.join(build_dir, "quality/pers/rerank_service/tools/event_log_dump/event_log_dump"),
        os.path.join(package_dir, "event_log_dump")
    )


def copy_daemon(sb_task, build_dir, package_dir, program):
    bin_path = "quality/pers/rerank_service/daemons/%s/%s" % (program, program)
    copy_path(
        os.path.join(build_dir, bin_path),
        os.path.join(package_dir, "atom-daemon")
    )
    if sb_task.ctx[SymbolsOptionalDumper.name]:
        gen_symdump(sb_task, package_dir, build_dir, bin_path, dump_file_name="atom-daemon.sym")


def copy_config(arcadia_tests_data_dir, package_dir, cfg_name):
    copy_path(
        os.path.join(arcadia_tests_data_dir, "pers/rerank_service/%s.json" % cfg_name),
        os.path.join(package_dir, "cfg.json")
    )


def copy_data(arcadia_tests_data_dir, package_dir, subpath, targpath=None):
    copy_path(
        os.path.join(arcadia_tests_data_dir, "pers/rerank_service", subpath),
        os.path.join(package_dir, subpath if targpath is None else targpath)
    )


def copy_resource_data(res_path, package_dir, subpath, targpath=None):
    copy_path(
        res_path,
        os.path.join(package_dir, subpath if targpath is None else targpath)
    )


class PkgBuilderBase(object):
    archive = True

    @abc.abstractmethod
    def get_pkg_name(self):
        pass

    @abc.abstractmethod
    def get_resource_type(self):
        pass

    @abc.abstractmethod
    def fill_package(self, task, build_dir, arcadia_tests_data_dir, res, package_dir):
        pass

    def get_required_subtasks(self):
        return []

    def get_required_resources(self):
        return []


class PersRerankdPkgBuilder(PkgBuilderBase):

    def __init__(self):
        self.cfg_name = "release-cfg"
        self.daemon_name = _PERS_RERANKD

        self.data = [
            ("../../wizard/language/pure", "pure")
        ]

        self.resource_data = [
            ("BUILD_PERS_RERANK_SERVICE_FMLS", "PERS_RERANK_SERVICE_FMLS", "fml"),
            ("BUILD_PERS_RERANK_SERVICE_DATA", "PERS_RERANK_SERVICE_DATA", "data")
        ]

    def fill_package(self, sb_task, build_dir, arcadia_tests_data_dir, res, package_dir):
        copy_daemon(sb_task, build_dir, package_dir, self.daemon_name)
        copy_event_log_dump(build_dir, package_dir)
        copy_config(arcadia_tests_data_dir, package_dir, self.cfg_name)
        for x in self.data:
            subpath = x[0] if isinstance(x, tuple) else x
            targpath = x[1] if isinstance(x, tuple) else None
            copy_data(arcadia_tests_data_dir, package_dir, subpath, targpath)

        if self.resource_data:
            for rd in self.resource_data:
                task_t, res_t, subpath = rd
                assert res_t in res, "Resource not found: " + res_t
                copy_resource_data(res[res_t], package_dir, subpath)

    def get_pkg_name(self):
        return "pers_ah_rerank_service_bundle"

    def get_resource_type(self):
        return resource_types.PERS_APPHOST_RERANK_SERVICE_BUNDLE

    def get_required_subtasks(self):
        return [res[0] for res in self.resource_data]

    def get_required_resources(self):
        return [res[1] for res in self.resource_data]


class AtomFrontBuilder(PkgBuilderBase):

    def _update_data(self, fname, dest):
        _UATRAITS_DATA_TEMPL = "https://github.yandex-team.ru/raw/InfraComponents/uatraits-data/master/data/%s"

        r = requests.get(_UATRAITS_DATA_TEMPL % fname)
        data = r.text
        if not data:
            raise Exception("Failed to fetch uatraits dict: " + fname)

        logging.info("Fetched %s, size %d" % (fname, len(data)))

        with open(dest, 'wb') as f:
            f.write(data.encode("utf-8"))

    def fill_package(self, sb_task, build_dir, arcadia_tests_data_dir, res, package_dir):
        copy_daemon(sb_task, build_dir, package_dir, "front")
        copy_event_log_dump(build_dir, package_dir)
        copy_path(
            os.path.join(arcadia_tests_data_dir, "pers/rerank_service/front/release-cfg.properties"),
            os.path.join(package_dir, "cfg.properties")
        )
        copy_data(arcadia_tests_data_dir, package_dir, "front/data", "data")
        self._update_data("browser.xml", os.path.join(package_dir, "data/browser.xml"))

    def get_pkg_name(self):
        return "pers_atom_front_service_bundle"

    def get_resource_type(self):
        return resource_types.PERS_ATOM_FRONT_SERVICE_BUNDLE


class ConfigBundleBuilder(PkgBuilderBase):
    INARCHIVE_FILES = ('push-client.yml',)

    def fill_package(self, sb_task, build_dir, arcadia_tests_data_dir, res, package_dir):
        logging.info("VARIABLES: {} {} {} {}".format(sb_task, build_dir, arcadia_tests_data_dir, package_dir))
        os.chdir(package_dir)
        result = process(
            build_dir=os.path.join(arcadia_tests_data_dir, 'pers/rerank_service/config/'),
            config=os.path.join(arcadia_tests_data_dir, "pers/rerank_service/front/release-cfg.properties")
        )
        logging.info("COPYING: {}".format(self.INARCHIVE_FILES))
        for inarchive_file in self.INARCHIVE_FILES:
            copy_path(
                os.path.join(arcadia_tests_data_dir, 'pers/rerank_service/config/{}'.format(inarchive_file)),
                os.path.join(package_dir, inarchive_file)
            )
        logging.info("RESULT: {}".format(result))

    def get_pkg_name(self):
        return "config"

    def get_resource_type(self):
        return resource_types.PERS_ATOM_CONFIG_BUNDLE


class ConfigBundleIssHookInstallBuilder(PkgBuilderBase):
    archive = False
    filename = "iss_hook_install"

    def fill_package(self, sb_task, build_dir, arcadia_tests_data_dir, res, package_dir):
        copy_path(
            os.path.join(arcadia_tests_data_dir, "pers/rerank_service/config/{}".format(self.filename)),
            os.path.join(package_dir, self.filename)
        )

    def get_pkg_name(self):
        return "hook"

    def get_resource_type(self):
        return resource_types.PERS_ATOM_CONFIG_ISS_HOOK_INSTALL


class PersRerankdProdLoopConf(PkgBuilderBase):
    archive = False
    filename = "instancectl.conf"

    def fill_package(self, sb_task, build_dir, arcadia_tests_data_dir, res, package_dir):
        copy_path(
            os.path.join(arcadia_tests_data_dir, "pers/rerank_service/{}".format(self.filename)),
            os.path.join(package_dir, self.filename)
        )

    def get_pkg_name(self):
        return "prod-loop"

    def get_resource_type(self):
        return resource_types.PERS_RERANK_SERVICE_INSTANCECTL_PROD_LOOP_CONF


class PersRerankdPushClientConf(PkgBuilderBase):
    archive = False
    filename = "push-client.yml"

    def fill_package(self, sb_task, build_dir, arcadia_tests_data_dir, res, package_dir):
        copy_path(
            os.path.join(arcadia_tests_data_dir, "pers/rerank_service/{}".format(self.filename)),
            os.path.join(package_dir, self.filename)
        )

    def get_pkg_name(self):
        return "rerankd-pushclient-conf"

    def get_resource_type(self):
        return PERS_RERANK_SERVICE_PUSHCLIENT_CONF


class RerankdDataUpdaterBuilder(PkgBuilderBase):
    archive = False
    filename = "data_updater.sh"

    def fill_package(self, sb_task, build_dir, arcadia_tests_data_dir, res, package_dir):
        copy_path(
            os.path.join(arcadia_tests_data_dir, "pers/rerank_service/{}".format(self.filename)),
            os.path.join(package_dir, self.filename)
        )

    def get_pkg_name(self):
        return "data-updater"

    def get_resource_type(self):
        return resource_types.PERS_ATOM_CONFIG_RERANKD_DATA_UPDATER


def _get_pkg_builder(daemon_name):
    return {
        _PERS_RERANKD: PersRerankdPkgBuilder(),
        _ATOM_FRONT: AtomFrontBuilder(),
        "config": ConfigBundleBuilder(),
        "hook": ConfigBundleIssHookInstallBuilder(),
        "prod-loop": PersRerankdProdLoopConf(),
        "rerankd-pushclient-conf": PersRerankdPushClientConf(),
        "data-updater": RerankdDataUpdaterBuilder()
    }[daemon_name]


_ATOM_RELEASE_VAULT_NAME = "atom-releases"
_ATOM_RELEASE_VAULT_OWNER = "robot-atom-releases"
_ARCADIA_DIRECTORY = "arcadia"
_ARCADIA_TEST_DIRECTORY = "arcadia_tests_data"
_ATOM_RELEASE_DIRECTORY = "arcadia:/arc/branches/atom/releases"
_ATOM_SERVICE_NAME = "Atom"


class BuildAtomDaemon(nanny.ReleaseToNannyTask, ArcadiaTask):

    """Builds pers-rerankd binary and packs it together with reranking formulas and config"""

    type = "BUILD_ATOM_DAEMON"

    execution_space = 100000

    input_parameters = build_params.get_build_system_params() + [
        SandboxArcadiaUrlParameter,
        SelectDaemon,
        SymbolsOptionalDumper,
        ShouldMakeNewRelease
    ]

    release_to = ["mobile-search-releases", "librarian", "qkrorlqr", "manokk"]
    _releaser = Releaser(_ATOM_SERVICE_NAME, _ATOM_RELEASE_VAULT_NAME, _ATOM_RELEASE_VAULT_OWNER, _ATOM_RELEASE_DIRECTORY)

    client_tags = ctc.Tag.Group.LINUX

    def do_execute(self):
        BuildAtomDaemon._releaser.set_task(self)
        BuildAtomDaemon._releaser.init_release_dir()

        daemon_name = self.ctx.get(SelectDaemon.name)
        daemons = _DAEMON_COMPONENTS.get(daemon_name, [daemon_name])
        builders = {name: _get_pkg_builder(name) for name in daemons}

        required_res_types = set()
        for builder in builders.itervalues():
            required_res_types.update(builder.get_required_resources())

        resources = {}
        # start dependent subtasks
        child_tasks_ids = self.ctx.get("child_tasks_ids", [])
        if not child_tasks_ids:
            subtask_types = set()
            for builder in builders.itervalues():
                subtask_types.update(builder.get_required_subtasks())

            for t in subtask_types:
                child_tasks_ids.append(
                    self.create_subtask(
                        t,
                        "Subtask for %s #%d" % (self.type, self.id)
                    ).id
                )

            if child_tasks_ids:
                self.ctx['child_tasks_ids'] = child_tasks_ids
                self.wait_all_tasks_completed(child_tasks_ids)

                # wait for tasks to finish
                return
        else:
            check_if_tasks_are_ok(child_tasks_ids)
            logging.info('All subtasks are ready')

            # fetching resources
            for task_id in child_tasks_ids:
                for res in channel.sandbox.list_resources(task_id=task_id):
                    if res.type in required_res_types and res.type not in resources:
                        resources[res.type] = self.sync_resource(res.id)

        # prepare build environment
        build_dir = self.abs_path("build")
        make_folder(build_dir)
        os.chdir(build_dir)

        arcadia_dir = self.get_arcadia_src_dir()

        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        test_data_path = parsed_url.path.rstrip("/")
        assert test_data_path.endswith(_ARCADIA_DIRECTORY)
        test_data_path = test_data_path[:-8] + "/arcadia_tests_data/"
        test_data_url = Arcadia.replace(url, path=test_data_path)

        BuildAtomDaemon._releaser.make_new_release_if_needed(url)

        arcadia_tests_data_dir = ArcadiaTestData.get_arcadia_test_data(self, test_data_url)
        targets = ["quality/pers/rerank_service"]

        build_type = consts.RELEASE_BUILD_TYPE
        sdk.do_build(
            self.ctx["build_system"], arcadia_dir, targets, build_type, clear_build=True,
            results_dir=build_dir
        )

        # create package
        processed_pkgs = set()
        for name, pkg_builder in builders.iteritems():
            pkg_version = parsed_url.revision or "HEAD"

            if pkg_builder.get_pkg_name() in processed_pkgs:
                raise Exception("duplicate pkg name: %s" % pkg_builder.get_pkg_name())
            processed_pkgs.add(pkg_builder.get_pkg_name())

            pkg_name = pkg_builder.get_pkg_name() + "_" + str(pkg_version)

            cur_dir = os.path.join(build_dir, pkg_name + ".tardir")
            make_folder(cur_dir)
            os.chdir(cur_dir)
            package_dir = os.path.join(cur_dir, pkg_name)
            make_folder(package_dir)

            pkg_builder.fill_package(
                self,
                build_dir,
                arcadia_tests_data_dir,
                resources,
                package_dir
            )
            if name == _PERS_RERANKD or name == _ATOM_FRONT:
                BuildAtomDaemon._releaser.add_release_file(package_dir)

            os.chdir(cur_dir)

            resource_dir = self.abs_path()

            if pkg_builder.archive:
                run_process("tar czvf %s.tar.gz -C %s ." % (pkg_name, pkg_name), shell=True)

                tar = glob.glob(os.path.join(cur_dir, "*.tar.gz"))[-1]

                # save created package as resource
                copy_path(tar, resource_dir)
                resource_name = os.path.basename(tar)

            else:
                copy_path(
                    os.path.join(package_dir, pkg_builder.filename),
                    os.path.join(resource_dir, pkg_builder.filename)
                )
                resource_name = os.path.basename(os.path.join(resource_dir, pkg_builder.filename))

            self.create_resource(
                description=resource_name,
                resource_path=os.path.join(resource_dir, resource_name),
                resource_type=pkg_builder.get_resource_type(),
                arch=self.arch,
            )


__Task__ = BuildAtomDaemon
