import tarfile
import ConfigParser
import os
import collections
import socket

from sandbox.sandboxsdk.paths import copy_path


def resolved_instances(f):
    data = {}

    def wrapped(value):
        if value in data.keys():
            pass
        else:
            data.setdefault(value, f(value))
        return data[value]
    return wrapped


@resolved_instances
def resolve_instances(value):
    from library.sky.hostresolver import Resolver

    value = value.split('%')[1]
    instances = ''
    data = Resolver().resolveInstances(value)
    for host in data:
        port = list(data[host])[0][1].split('@')[0].split(':')[1]
        instance = host + ':' + port
        instances += instance + ','
    return instances


class MobreportParseIniData(object):

    def __init__(self, build_dir, ini_dir, instances_file='instances.ini', templates_file='templates.ini'):
        self.ini_dir = os.path.join(build_dir, ini_dir)
        self.instances_file = instances_file
        self.templates_file = templates_file
        self.parse_separator = '_'
        self.template_section = 'template_'
        self.load_configs()

    def instances_file_path(self):
        return os.path.join(
            self.ini_dir,
            self.instances_file
        )

    def templates_file_path(self):
        return os.path.join(
            self.ini_dir,
            self.templates_file
        )

    def read_ini_from_dir(self):
        files = []
        for filename in os.listdir(self.ini_dir):
            if filename.endswith('ini') \
                    and not filename == self.instances_file \
                    and not filename == self.templates_file:
                files.append(os.path.join(self.ini_dir, filename))

        return files

    def load_configs(self):
        self.configs = ConfigParser.RawConfigParser(allow_no_value=True)
        self.configs.optionxform = str
        self.configs.read(self.read_ini_from_dir())

        self.templates = ConfigParser.RawConfigParser(allow_no_value=True)
        self.templates.optionxform = str
        self.templates.read(self.templates_file_path())

        self.instances = ConfigParser.RawConfigParser(allow_no_value=True)
        self.instances.optionxform = str
        self.instances.read(self.instances_file_path())

    def _parse_section(self, instance, inst_data):
        for section in instance.split(self.parse_separator):
            if self.configs.has_section(section):
                inst_data.update(self.configs.items(section))

    def _parse_section_options(self, instances, instance, inst_data):
        options = instances.options(instance)
        if options:
            self._parse_section(options[0], inst_data)

    def parse_ini(self):
        templates = self._parse_data_from_ini(
            self.templates,
            {},
            istemplate=True
        )
        instances = self._parse_data_from_ini(
            self.instances,
            templates,
            istemplate=False
        )
        return instances

    def _parse_data_from_ini(self, instances, templates, istemplate):
        data = {}
        for instance in instances.sections():
            inst_data = collections.OrderedDict()
            ctype = instance.split(self.parse_separator)[1]
            template_name = self.template_section + ctype

            if not istemplate:
                if template_name in templates.keys():
                    inst_data.update(
                        templates[self.template_section + ctype]
                    )
                self._parse_section(instance, inst_data)

            self._parse_section_options(instances, instance, inst_data)
            data.setdefault(instance, inst_data)

        return data


class MobreportConfigs(MobreportParseIniData):

    def __init__(self, build_dir, ini_dir, file_ext):
        self.file_ext = file_ext
        MobreportParseIniData.__init__(
            self,
            build_dir,
            ini_dir,
            'instances.ini',
            'templates.ini'
        )

    def init_file(self, instance):
        return "%s.%s" % (instance, self.file_ext)

    def init_file_data(self, instance):
        return ''

    def generate_files(self, data):
        fnames = []
        for instance in data:
            fname = self.init_file(instance)
            fdata = self.init_file_data(instance)
            for k, v in data[instance].iteritems():
                if v is None:
                    fdata += "%s\n" % k
                else:
                    fdata += "%s=%s\n" % (k, v)

            with open(fname, 'w') as f:
                f.write(fdata)
            fnames.append(fname)

        return fnames

    def create_inidata(self, inidata):
        data = self.parse_ini()
        data2 = {}
        for instance in data:
            inst_data = collections.OrderedDict()
            if inidata:
                inst_data.update(inidata)
            inst_data.update(data[instance])
            data2.setdefault(instance, inst_data)

        return data2

    def create_files(self, inidata=None):
        return self.generate_files(
            self.create_inidata(inidata)
        )


class MobreportStartupFiles(MobreportConfigs):

    def __init__(self, build_dir):
        MobreportConfigs.__init__(self, build_dir, 'ini/startup', 'sh')

    def init_file_data(self, instance):
        return "instance_properties=%s.properties\n" % instance


class MobreportInstancesFiles(MobreportConfigs):

    def __init__(self, build_dir):
        MobreportConfigs.__init__(self, build_dir, 'ini/configs', 'properties')

    def create_files(self, inidata):
        inidata = self.parse_instances_hosts_data(
            self.create_inidata(
                self.get_report_config_data(inidata)
            )
        )
        MobreportDnsCache().create_cache(inidata)
        return self.generate_files(inidata)

    @staticmethod
    def parse_instances_hosts_data(inidata):
        for data in inidata:
            for item in inidata[data]:
                v = inidata[data][item]
                if item.endswith('.host') and v.startswith('SKYLI%'):
                    inidata[data][item] = \
                        resolve_instances(v)

        return inidata

    @staticmethod
    def get_report_config_data(path):
        config_data = collections.OrderedDict()
        if not os.path.isfile(path):
            return config_data

        with open(path, 'r') as f:
            for line in f:
                # noinspection PyBroadException
                try:
                    k, v = line.rstrip().split('=', 1)
                except:
                    pass
                else:
                    config_data[k.strip()] = v.strip()
            return config_data


class MobreportDnsCache(object):

    def __init__(self):
        self.tlds = ['.ru', '.com', '.com.tr', '.ua', '.by', '.kz']
        self.dnscacheconf = 'dnscache.conf'

    def create_cache(self, inidata):
        return self.generate_files(inidata)

    def generate_files(self, data):
        hosts = []
        for v in data.itervalues():
            for k, v1 in v.iteritems():
                if k.endswith('.host'):
                    self._parse_host(hosts, v1)

        return self._resolve_hosts(set(hosts))

    def _parse_host(self, hosts, rawhost):
        rawhosts = rawhost.split(',')
        for rawhost in rawhosts:
            rawhost = rawhost.replace('\\', '').split(':')[0]
            if rawhost.endswith('{domain}'):
                host = rawhost.replace('.{domain}', '')
                if host.startswith('{news_host}'):
                    host = host.replace('{news_host}.', '')
                    for domain in self.tlds:
                        fqdn = 'news.' + host + domain
                        hosts.append(fqdn)
                else:
                    for domain in self.tlds:
                        fqdn = host + domain
                        hosts.append(fqdn)
            else:
                if rawhost == 'localhost':
                    pass
                else:
                    hosts.append(rawhost)

    def _resolve_hosts(self, hosts):
        data = {}
        for host in hosts:
            # noinspection PyBroadException
            try:
                res_data = socket.getaddrinfo(
                    host, 80, 0, 0,
                    socket.IPPROTO_TCP
                )
            except:
                pass
            else:
                data.setdefault(host, {})
                for t in res_data:
                    data[host].setdefault('ips', [])
                    data[host]['ips'].append(t[4][0])

        return self._write_hosts_to_file(data)

    def _write_hosts_to_file(self, hosts):
        fnames = []
        hosts_data = self._hosts_to_fdata(hosts)
        with open(self.dnscacheconf, 'w') as f:
            f.write(hosts_data)
        fnames.append(self.dnscacheconf)
        return fnames

    @staticmethod
    def _hosts_to_fdata(hosts):
        fdata = '<DNSCache>\nDNSCache '
        for host in hosts:
            for ip in hosts[host]['ips']:
                fdata = fdata + \
                    host + '=' + ip + ' '
        fdata += '\n</DNSCache>\n'
        return fdata


def create_tar(build_dir, package_dir, filenames):
    tar = tarfile.open(os.path.join(package_dir, 'properties.tgz'), 'w:gz')
    for fname in filenames:
        if fname in Resources.inarchive:
            tar.add(os.path.join(build_dir, fname), fname)
        else:
            tar.add(fname)
            os.remove(fname)
    tar.close()


def copy_files(build_dir, package_dir, filenames):
    for fname in filenames:
        copy_path(os.path.join(build_dir, fname), os.path.join(package_dir, fname))


def inarchive_resources():
    fnames = []
    for filename in Resources.inarchive:
        fnames.append(filename)
    return fnames


class Resources(object):
    outarchive = [
        ('MOBILESEARCH_REPORT_ISS_HOOK_INSTALL',
         'iss_hook_install'),
    ]

    inarchive = [
        'push-client.yml'
    ]


class MobreportLoopFiles():

    def __init__(self, build_dir):
        self.ini_dir = os.path.join(build_dir, 'ini/loop')
        self.file_ext = 'conf'
        self.instances_file = 'instances.ini'
        self.templates_file = 'templates.ini'
        self.parse_separator = '_'
        self.section_name = "section_name"
        self.template_section = 'template_'

    def instances_file_path(self):
        return os.path.join(
            self.ini_dir,
            self.instances_file
        )

    def templates_file_path(self):
        return os.path.join(
            self.ini_dir,
            self.templates_file
        )

    def read_ini_from_dir(self):
        files = []
        for filename in os.listdir(self.ini_dir):
            if filename.endswith('ini') \
                    and not filename == self.instances_file \
                    and not filename == self.templates_file:
                files.append(os.path.join(self.ini_dir, filename))

        return files

    def load_configs(self):
        self.configs = ConfigParser.RawConfigParser(allow_no_value=True)
        self.configs.optionxform = str
        self.configs.read(self.read_ini_from_dir())

        self.templates = ConfigParser.RawConfigParser(allow_no_value=True)
        self.templates.optionxform = str
        self.templates.read(self.templates_file_path())

        self.instances = ConfigParser.RawConfigParser(allow_no_value=True)
        self.instances.optionxform = str
        self.instances.read(self.instances_file_path())

    def _parse_templates(self):
        # parse template sections and produce dictionary of lists
        # of sections
        self.load_configs()
        instances_sections = {}
        for instance in self.templates.sections():
            ctype = instance.split(self.parse_separator)[1]
            instances_sections.setdefault(ctype, self.templates.options(instance)[0].split(self.parse_separator))
        return instances_sections

    def _parse_instances(self):
        # parse instances sections and produce dictionary of lists
        # of sections if any appending them to the result of _parse_templates
        templates_sections = self._parse_templates()
        instances_sections = {}

        for instance in self.instances.sections():
            ctype = instance.split(self.parse_separator)[1]
            section_options = []
            template_options = []
            template_options = templates_sections[ctype]

            if self.instances.options(instance):
                section_options = template_options + self.instances.options(instance)[0].split(self.parse_separator)
            else:
                section_options = template_options

            instances_sections.setdefault(instance, section_options)

        return instances_sections

    def _prepare_configs(self):
        # get result of _parse_instances and read all the sections
        # from self.configs, applying them depending on rules
        # if there is two sections with identical section_name
        # for exampe 'mobreport' (sections mobreport-iss-prod,
        # mobreport-bsconfig-prod) and option name looks like
        # mobreport-iss-prod_mobreport-bsconfig-prod
        # there will be produced section [mobreport] with all
        # options from mobreport-iss-prod and updated with options
        # from mobreport-bsconfig-prod
        instances = self._parse_instances()

        # prepare dict with key-value: section name in settings => section_name
        sections_pair = {}
        for section in self.configs.sections():
            if self.configs.has_option(section, self.section_name):
                sections_pair[section] = self.configs.get(section, self.section_name)

        # prepare dict instance->ConfigParser object
        instances_pair = {}
        for (instance, sections) in instances.items():
            self.instance_config = ConfigParser.RawConfigParser(allow_no_value=True)
            self.instance_config.optionxform = str

            for section in sections:
                sname = sections_pair[section]
                if not self.instance_config.has_section(sname):
                    self.instance_config.add_section(sname)

                for option in self.configs.options(section):
                    if option not in [self.section_name]:
                        self.instance_config.set(sname, option, self.configs.get(section, option))

            instances_pair[instance] = self.instance_config

        return instances_pair

    def create_files(self):
        filenames = []
        configs = self._prepare_configs()

        for (instance, config) in configs.items():
            filename = "{instance}.{ext}".format(instance=instance, ext=self.file_ext)
            with open(filename, "w+") as fp:
                config.write(fp)
                filenames.append(filename)
                fp.close()

        return filenames


def process(build_dir='./', config='mobilesearch_report_atom_cfg.properties'):

    return (
        MobreportStartupFiles(build_dir).create_files() +
        MobreportInstancesFiles(build_dir).create_files(config) +
        MobreportLoopFiles(build_dir).create_files() +
        inarchive_resources() +
        ['dnscache.conf']
    )


if __name__ == "__main__":
    process()
