# coding: utf-8

import os
import tarfile

import sandbox.common.types.misc as ctm
from sandbox.projects import resource_types
from sandbox.projects.common.apihelpers import get_last_resource
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox import sdk2


class AURORA_PARSER_LAYER(sdk2.resource.AbstractResource):
    """
        Resource with layer for aurora parser
    """
    releasable = False
    any_arch = False
    auto_backup = True


class AURORA_ABSTRACT_RESOURCE(sdk2.resource.AbstractResource):
    """
        Aurora abstract resource
    """
    releasable = False
    any_arch = False
    auto_backup = True


class AuroraBundleSvnRevision(SandboxStringParameter):
    name = 'revision'
    description = 'Aurora bundle svn revision'
    default_value = None


class AuroraBundleSvnUrl(SandboxStringParameter):
    name = 'svn_url'
    description = 'Svn url for aurora source code'
    default_value = os.path.join(Arcadia.trunk_url(), 'aurora')


class BuildAuroraBundle(SandboxTask):

    type = 'BUILD_AURORA_BUNDLE'

    dns = ctm.DnsType.DNS64

    owners = ['scomip']

    input_parameters = [AuroraBundleSvnRevision, AuroraBundleSvnUrl]

    resource_path = 'aurora-bundle.tar.gz'
    resource_type = resource_types.AURORA_BUNDLE_VENV

    _checkout_path = 'src'

    @staticmethod
    def _run_process(cmd, log_prefix, **kwargs):
        env = os.environ.copy()
        env['PYTHONHOME'] = os.getcwd()
        return run_process(cmd, shell=True, environment=env, log_prefix=log_prefix, **kwargs)

    @property
    def aurora_src(self):
        return self.path(self._checkout_path)

    def get_version(self):
        proc = self._run_process(['../bin/python', 'setup.py', '--version'], 'get_version', work_dir=self.aurora_src,
                                 outs_to_pipe=True)
        return proc.communicate()[0].strip()

    def on_prepare(self):
        revision, url = self.ctx[AuroraBundleSvnRevision.name], self.ctx[AuroraBundleSvnUrl.name]

        Arcadia.checkout(url, self.aurora_src, revision=revision)

        bundle_info = {
            'revision': revision,
            'build': 'trunk' if 'trunk' in url else url.split('/')[-1],
            'sandbox_task_id': self.id,
        }
        with open(os.path.join(self.aurora_src, 'aurora', 'config', 'bundle.py'), 'w') as bundle_file:
            bundle_file.write('BUNDLE_BUILD_INFO = {}'.format(repr(bundle_info)))

        # Get python bundle to build needed wheels
        python_bundle_archive = self.sync_resource(get_last_resource(resource_types.AURORA_PYTHON_BUNDLE))
        with tarfile.open(python_bundle_archive) as tar:
            tar.extractall()

    def on_execute(self):
        """
            Build a bundle of Aurora project with custom python
        """

        # Install pip
        self._run_process(['curl', 'https://bootstrap.pypa.io/get-pip.py', '|', 'bin/python', '-'],
                          'install_pip')
        # Install six
        self._run_process(['bin/python', 'bin/pip', 'install', '-i', 'https://pypi.python.org/simple',
                           '--extra-index-url', 'https://pypi.yandex-team.ru/simple', '--retries', '10', 'six'],
                          'install_six')
        # Install requirements
        self._run_process(['bin/python', 'bin/pip', 'install', '-i', 'https://pypi.python.org/simple',
                           '--extra-index-url', 'https://pypi.yandex-team.ru/simple', '--retries', '10',
                           self.aurora_src],
                          'install_aurora')

        # Install pyrasite for debug
        self._run_process(['bin/python', 'bin/pip', 'install', '-i', 'https://pypi.python.org/simple',
                           '--extra-index-url', 'https://pypi.yandex-team.ru/simple', '--retries', '10', 'pyrasite'],
                          'install_pyrasite')

        # get version from installed package
        version = self.get_version()

        with tarfile.open(self.resource_path, 'w:gz') as tar:
            for _ in ['bin', 'lib', 'include']:
                tar.add(os.path.join('.', _), arcname=_)
            for _ in ('scripts', 'xpath', 'download_utils', 'ci'):
                tar.add(os.path.join(self.aurora_src, _), arcname=_)

        self.create_resource(
            'Venv with Aurora',
            resource_path=self.resource_path,
            resource_type=self.resource_type,
            arch='linux',
            attributes={
               'version': version,
               'ttl': 365,
            }
        )


__Task__ = BuildAuroraBundle
