# coding: utf-8
import logging
import os
import tarfile
import time

import sandbox.common.types.task as ctt
from sandbox import common
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.balancer import resources as balancer_resources
from sandbox.projects.common import environments, apihelpers
from sandbox.projects.nanny.common.gobabygo_parameters import GitRefIdParameter, GitRefShaParameter, ReleaseParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common.nanny import nanny


class BalancerExecutable(LastReleasedResource):
    name = 'balancer_executable_resource_id'
    description = 'Balancer executable resource id'
    resource_type = [balancer_resources.BALANCER_EXECUTABLE]

    @common.utils.classproperty
    def default_value(cls):
        stable_balancer = apihelpers.get_last_released_resource(balancer_resources.BALANCER_EXECUTABLE, arch='linux',
                                                                release_status=ctt.ReleaseStatus.STABLE)
        testing_balancer = apihelpers.get_last_released_resource(balancer_resources.BALANCER_EXECUTABLE, arch='linux',
                                                                 release_status=ctt.ReleaseStatus.TESTING)
        return max(stable_balancer.id if stable_balancer else None,
                   testing_balancer.id if testing_balancer else None)


class BuildAwacs(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Задача тестирования и сборки awacs.
        При сборке создаётся пустой venv, в него устанавливается пакет awacs и
        все его зависимости.
    """
    type = 'BUILD_AWACS'

    client_tags = Tag.LINUX_XENIAL
    cores = 15
    required_ram = 16 * 1024

    input_parameters = [
        GitRefIdParameter,
        GitRefShaParameter,
        ReleaseParameter,
        BalancerExecutable,
        nanny.StartrekTicketIdsParameter
    ]

    environment = [
        environments.SandboxGitEnvironment('2.19.0'),
        environments.SandboxJavaJdkEnvironment('1.8.0'),
        environments.SwatZookeeperEnvironment('3.4.6'),
        environments.SwatMongoDbEnvironment('3.4.2'),
    ]

    GIT_URL = 'https://{username}:{password}@bb.yandex-team.ru/scm/nanny/nanny.git'
    CHECKOUT_PATH = 'nanny'
    TGZ_PATH = 'awacs.tar.gz'
    PIP_TGZ = 'pip-20.3.4.tar.gz'

    def _get_awacs_src_dir(self):
        return os.path.join(self.CHECKOUT_PATH, 'awacs')

    def _get_venv_path(self):
        return self.path('awacs_build')

    def _get_venv_bin_path(self):
        return os.path.join(self._get_venv_path(), 'bin')

    def _get_venv_python_path(self):
        return os.path.join(self._get_venv_bin_path(), 'python')

    def _run_pip_install(self, args, log_prefix=None):
        return run_process([self._get_venv_python_path(), '-m', 'pip', 'install'] + args,
                           log_prefix=log_prefix,
                           work_dir=self._get_awacs_src_dir())

    def _get_checkout_url(self):
        oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
        return self.GIT_URL.format(username='x-oauth-token', password=oauth_token)

    @staticmethod
    def _run_git_command(command, args, work_dir=None):
        run_process(
            ['git', command] + args,
            log_prefix='git_%s' % command,
            work_dir=work_dir,
            shell=True
        )

    def _checkout(self, url, dest, ref=None):
        if ref is None:
            self._run_git_command('clone', [url, dest, '--depth', '1'])
        else:
            self._run_git_command('init', [dest])
            self._run_git_command('remote', ['add', 'origin', url], work_dir=dest)
            self._run_git_command('fetch', ['origin', ref, '--depth', '1'], work_dir=dest)
            self._run_git_command('reset', ['--hard', 'FETCH_HEAD'], work_dir=dest)

    def _install_packages(self):
        # update pip to be able to install wheels
        self._run_pip_install(['-U', self.PIP_TGZ])

        # install dependencies from local dir, fail if some are not satisfied
        self._run_pip_install(
            ['--no-index', '--find-links=./wheels', '-r', 'pip-requirements.txt'],
            log_prefix='awacs_reqs_pip_install'
        )
        # now install awacs itself
        self._run_pip_install(
            ['--no-index', '.'],
            log_prefix='awacs_pip_install',
        )

    def _make_resource(self, virtualenv_path, description):
        logging.info('creating tgz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
        self.create_resource(
            description=description,
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.AWACS,
            arch='linux'
        )

    def _set_awacs_version(self):
        awacs_src_dir = self._get_awacs_src_dir()
        version_file_path = os.path.join('src', 'awacs', 'version.py')
        run_process(['sed', '-i',
                     '"s/development_version/$(/skynet/python/bin/python setup.py --version)/"', version_file_path],
                    shell=True,
                    work_dir=awacs_src_dir)
        now = str(int(time.time()))
        run_process(['sed', '-i', '"s/TIMESTAMP = -1/TIMESTAMP = {}/"'.format(now), version_file_path],
                    shell=True,
                    work_dir=awacs_src_dir)

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def _run_tests(self):
        venv_bin_path = self._get_venv_bin_path()
        awacs_src_dir = self._get_awacs_src_dir()

        self._run_pip_install(
            ['--no-index', '--find-links=./wheels', '-r', 'pip-dev-requirements.txt'],
            log_prefix='awacs_dev_reqs_pip_install'
        )

        env = dict(os.environ)
        path_parts = [venv_bin_path]
        if 'PATH' in env:
            path_parts.append(env['PATH'])
        env_path = ':'.join(path_parts)
        env.update({'PATH': env_path})

        balancer = channel.task.sync_resource(self.ctx[BalancerExecutable.name])

        old_do_not_restart = self.ctx['do_not_restart']
        self.ctx['do_not_restart'] = True
        run_process([
            './test-all.sh',
            'tests',
            '--zookeeper={}'.format(os.environ['ZOOKEEPER_DIR']),
            '--balancer={}'.format(balancer),
        ], environment=env, log_prefix='test_sh', work_dir=awacs_src_dir)
        self.ctx['do_not_restart'] = old_do_not_restart

    def on_execute(self):
        logging.info('checking out the source code')
        ref_id = self.ctx['ref_id']
        ref_sha = self.ctx.get('ref_sha')
        checkout_url = self._get_checkout_url()
        self._checkout(checkout_url, self.CHECKOUT_PATH, ref=ref_sha or ref_id)

        venv_path = self._get_venv_path()
        # create virtualenv using *skynet* python
        run_process(['/skynet/python/bin/virtualenv', venv_path])

        self._set_awacs_version()
        self._install_packages()

        logging.info('run tests')
        self._run_tests()

        if self.ctx.get('release'):
            run_process(['/skynet/python/bin/virtualenv', '--relocatable', venv_path])
            if ref_id.startswith('refs/tags/'):
                tag = ref_id[len('refs/tags/'):]
            else:
                tag = 'master'
            description = 'awacs virtualenv tgz {0}'.format(tag)
            self._make_resource(venv_path, description)

            oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
            run_process([self._get_venv_python_path(), './awacs/docs/upload_to_wiki.py', oauth_token],
                        log_prefix='upload_to_wiki', work_dir=self.CHECKOUT_PATH)


__Task__ = BuildAwacs
