# -*- coding: utf-8 -*-

import os
import logging

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox import common
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build import parameters as ya_make_parameters
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.utils import svn_last_change
from sandbox.projects.resource_types import BASS_BIN, BASS_CONFIG, BASS_PIPE_LOG_PARSER
from sandbox.sandboxsdk.parameters import SandboxRadioParameter
from sandbox.sandboxsdk.task import SandboxTask

logger = logging.getLogger('BuildBAssTask')


class ReleaseTypeParam(SandboxRadioParameter):
    name = 'release_type'
    description = 'Release type'
    choices = [
        ('production', 'production'),
        ('testing', 'testing'),
        ('rc', 'rc'),
    ]
    required = True
    group = 'Release type'


class BuildBAssTask(nanny.ReleaseToNannyTask, YaMakeTask):
    """
        Build BAss binary
    """
    client_tags = ctc.Tag.SSD & ctc.Tag.LINUX_XENIAL

    type = 'BUILD_BASS'

    input_parameters = [
        ya_make_parameters.ArcadiaUrl,
        ya_make_parameters.ClearBuild,
        ya_make_parameters.StripBinaries,
        ya_make_parameters.TestParameter,
        ya_make_parameters.TestCustomParameters,
        ya_make_parameters.TestTag,
        ya_make_parameters.UseArcadiaApiFuse,
        ya_make_parameters.Sanitize,
        ya_make_parameters.DefinitionFlags,
        ya_make_parameters.CheckReturnCode,
        ya_make_parameters.KeepOn,
        ya_make_parameters.YaTimeout,
        ReleaseTypeParam
    ]

    execution_space = 160 * 1024  # 160Gb will be enough
    required_ram = 4 * 1024

    target_path = 'alice/bass'
    rev_attr = 'rev'

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx['result_single_file'] = False
        self.ctx[consts.BUILD_BUNDLE_KEY] = False
        self.ctx['result_rd'] = 'Backend Assistant Bundle'
        logger.info('context: %s', self.ctx)

    def get_build_type(self):
        return consts.RELEASE_BUILD_TYPE

    def get_resources(self):
        return {
            BASS_BIN.name: {
                'description': BASS_BIN.name,
                'resource_type': BASS_BIN,
                'resource_path': 'bass_server'
            },
            BASS_CONFIG.name: {
                'description': BASS_CONFIG.name,
                'resource_type': BASS_CONFIG,
                'resource_path': 'config'
            },
            BASS_PIPE_LOG_PARSER.name: {
                'description': BASS_PIPE_LOG_PARSER.name,
                'resource_type': BASS_PIPE_LOG_PARSER,
                'resource_path': 'scripts'
            }
        }

    def get_targets(self):
        return [self.target_path]

    def get_arts(self):
        return [{'path': '{}/bin/bass_server'.format(self.target_path),
                 'dest': ''}]

    def get_arts_source(self):
        return [{'path': '{}/configs/production_config.json'.format(self.target_path),
                 'dest': 'config'},
                {'path': '{}/configs/testing_config.json'.format(self.target_path),
                 'dest': 'config'},
                {'path': '{}/configs/rc_config.json'.format(self.target_path),
                 'dest': 'config'},
                {'path': '{}/configs/hamster_config.json'.format(self.target_path),
                 'dest': 'config'},
                {'path': '{}/scripts/push_client_log_parsing.py'.format(self.target_path),
                 'dest': 'scripts'}]

    def get_resources_attrs(self):
        if self.rev_attr in self.ctx:
            release_params = self.get_release_attributes()
            return {BASS_BIN.name: release_params,
                    BASS_CONFIG.name: release_params,
                    BASS_PIPE_LOG_PARSER.name: release_params}
        return {}

    def on_execute(self):
        # use revision from URL or get last from SVN
        parts = self.ctx[consts.ARCADIA_URL_KEY].split('@')
        if len(parts) == 2:
            self.ctx[self.rev_attr] = parts[1]
        else:
            self.set_rev(self.get_last_rev())
        logger.info('on_execute(), context = {}'.format(self.ctx))

        os.environ['VAULT_TOKEN'] = sdk2.Vault.data('BASS', 'robot-bassist_vault_token')
        return YaMakeTask.on_execute(self)

    def on_release(self, additional_parameters):
        is_prod = additional_parameters['release_status'] in ('stable', 'prestable')
        sanitize = self.ctx[ya_make_parameters.Sanitize.name]

        if is_prod and sanitize:
            message = 'Failed to release task: tried to deploy sanitizer binaries to production!'
            self.set_info(message)
            raise common.errors.ReleaseError(message)

        if is_prod and self.ctx[ReleaseTypeParam.name] != 'production':
            message = 'Failed to release task: tried to deploy testing configs to production!'
            self.set_info(message)
            raise common.errors.ReleaseError(message)

        logger.info('on_release(), context = {}, additional_parameters = {}'.format(self.ctx, additional_parameters))
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

    def get_last_rev(self):
        if self.rev_attr in self.ctx:
            return self.ctx[self.rev_attr]
        url = '{}/{}'.format(self.ctx[consts.ARCADIA_URL_KEY], self.target_path)
        logger.info('get revision for {}'.format(url))
        return svn_last_change(url)

    def set_rev(self, rev):
        if self.rev_attr not in self.ctx:
            self.ctx[self.rev_attr] = rev
            self.ctx[consts.ARCADIA_URL_KEY] += '@' + rev

    def get_release_attributes(self, rev=None, rel=None, san=None):
        return {
            self.rev_attr: rev or self.ctx[self.rev_attr],
            'rel': rel or self.ctx[ReleaseTypeParam.name],
            'san': san or self.ctx[ya_make_parameters.Sanitize.name]
        }


__Task__ = BuildBAssTask
