# -*- coding: utf-8 -*-

import os
import logging

from sandbox.sandboxsdk.channel import channel

import sandbox.sandboxsdk.parameters as sb_params
import sandbox.projects.common.constants as consts
import sandbox.sdk2.parameters as parameters
import sandbox.projects.common.build.parameters as build_params
from sandbox.projects import resource_types
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.wizard import utils as wizard_utils

from sandbox.projects.websearch.begemot import AllBegemotServices
import sandbox.projects.websearch.begemot.resources as br


def _Resources(_res_={}):
    if not _res_:
        _res_.update({r.name: (label, r) for label, r in (
            ('eventlog dump tool w/ begemot events', br.BEGEMOT_EVLOGDUMP),
            ('eventlog stat tool w/ begemot events', br.BEGEMOT_EVLOGSTAT),
            ('tool to upload begemot logs to yt', br.BEGEMOT_EVLOG_UPLOADER),
            ('apphost context format conversion tool', br.BEGEMOT_AH_CONVERTER),
            ('unpackrichtree tool', resource_types.UNPACKRICHTREE),
            ('unpackreqbundle tool', resource_types.UNPACKREQBUNDLE),
            ('eventlog mapper', br.BEGEMOT_YT_EVENTLOG_MAPPER),
            ('begemot mapper', br.BEGEMOT_YT_MAPPER),
            ('begemot fastbuild rules downloader', br.BEGEMOT_FAST_BUILD_DOWNLOADER),
            ('begemot response-parser tool', br.BEGEMOT_RESPONSE_PARSER),
            ('begemot argument-parser tool', br.BEGEMOT_ARGUMENTS_PARSER),
            ('begemot cypress shard updater', br.BEGEMOT_SHARD_UPDATER),
        )})

        # This is temporary dirty hack
        _res_.update({
            'evlogdump': ('eventlog dump tool w/ begemot events', br.BEGEMOT_EVLOGDUMP),
            'evlogstat': ('eventlog stat tool w/ begemot events', br.BEGEMOT_EVLOGSTAT),
            'shard_updater': ('begemot cypress shard updater', br.BEGEMOT_SHARD_UPDATER),
        })
        # The end of temporary dirty hack

        binaries = {shard.binary_resource for shard in AllBegemotServices.values()}
        _res_.update({
            res.name: (os.path.basename(res.arcadia_build_path), res)
            for res in binaries
        })

    return _res_


class ClearBuild(sb_params.SandboxBoolParameter):
    name = consts.CLEAR_BUILD_KEY
    description = 'Clear build'
    default_value = False
    group = 'Base build params'


class _BuildTestingBinaries(sb_params.SandboxBoolParameter):
    '''Build binaries required by tests in addition to selected targets'''
    group = 'Targets selection'
    name = 'want_testing_utilities'
    targets = [
        br.BEGEMOT_YT_MAPPER, br.BEGEMOT_AH_CONVERTER, br.BEGEMOT_EVLOGDUMP,
        br.BEGEMOT_EVLOGSTAT, br.BEGEMOT_RESPONSE_PARSER,
        br.BEGEMOT_SHARD_UPDATER, resource_types.UNPACKRICHTREE, resource_types.UNPACKREQBUNDLE
    ]
    description = 'Build %s' % ', '.join(sorted([os.path.basename(t.arcadia_build_path) for t in targets]))
    default_value = False


class _BuildReleaseUtilityBinaries(sb_params.SandboxBoolParameter):
    '''Build binaries used in nanny service in addition to selected targets'''
    group = 'Targets selection'
    name = 'want_release_utilities'
    targets = [br.BEGEMOT_EVLOGDUMP, br.BEGEMOT_EVLOG_UPLOADER, br.BEGEMOT_ARGUMENTS_PARSER]
    description = 'Build %s' % ', '.join(sorted([os.path.basename(t.arcadia_build_path) for t in targets]))
    default_value = False


class Targets(parameters.CheckGroup):
    group = 'Targets selection'
    description = "Begemot executables to build"
    name = 'targets'
    choices = sorted([(value[0], key) for (key, value) in _Resources().iteritems()])
    default_value = []


class DryRun(sb_params.SandboxBoolParameter):
    group = 'Debug options'
    description = "Do nothing, should be used for testing"
    name = 'dry_run'
    default_value = False


class BuildSystem(parameters.String):
    name = consts.BUILD_SYSTEM_KEY
    description = 'Build system'
    required = True
    default_value = consts.SEMI_DISTBUILD_BUILD_SYSTEM
    group = 'Build system params'
    choices = [
        ('Ya', consts.YMAKE_BUILD_SYSTEM),
        # TODO: remove after experiments
        ('Ya force', consts.YA_MAKE_FORCE_BUILD_SYSTEM),
        ('semi-distbuild', consts.SEMI_DISTBUILD_BUILD_SYSTEM),
        ('distbuild', consts.DISTBUILD_BUILD_SYSTEM),
    ]
    sub_fields = {
        consts.YMAKE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
        consts.YA_MAKE_FORCE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
    }


class DefaultTrueMusl(parameters.Bool):
    name = 'begemot_musl'
    description = 'Build with musl-libc'
    default_value = True
    group = 'Build system params'


def _resource_by_name(name):
    try:
        return br.__dict__[name]
    except KeyError:
        return resource_types.__dict__[name]


def _build_path(arc_path):
    parts = arc_path.rsplit('/', 2)
    if len(parts) < 3:  # arcadia_build_path now always points to a binary, not a folder
        return arc_path
    return os.path.join(parts[0], parts[1])


class BuildBegemotExecutable(YaMakeTask):
    type = 'BUILD_BEGEMOT_EXECUTABLE'
    _targets_collected = 'targets_collected'
    execution_space = 150 * 1024  # https://sandbox.yandex-team.ru/task/271147239/view (over limit)
    required_ram = 50 * 1024  # https://sandbox.yandex-team.ru/task/848883507/view (building tensorflow failed again)
    TIMEOUT = 6 * 3600
    input_parameters = (
        Targets,
        _BuildReleaseUtilityBinaries,
        _BuildTestingBinaries,
        ClearBuild,
        build_params.ArcadiaUrl,
        build_params.ArcadiaPatch,
        BuildSystem,
        build_params.BuildType,
        build_params.Sanitize,
        DefaultTrueMusl,
        build_params.SeparateDebug,
        build_params.UseArcadiaApiFuse,
        build_params.YtStore,
        build_params.LTO,
        build_params.ThinLTO,
        build_params.DefinitionFlags,
        DryRun
    )

    # Here we need to list all possible tags and later refine them in on_enqueue.
    # That's the only possible way in SDK1 (and YaMakeTask is currently on SDK1).
    client_tags = wizard_utils.ALL_SANDBOX_HOSTS_TAGS
    cores = 24

    def on_enqueue(self):
        self.ctx['want_evlog_uploader'] = True
        self.ctx['aapi_fallback'] = True  # DEVTOOLS-4664

        targets = set(self.ctx.get(Targets.name, Targets.default_value))
        for t in [_BuildReleaseUtilityBinaries, _BuildTestingBinaries]:
            if self.ctx.get(t.name, t.default_value):
                targets |= set([r.name for r in t.targets])
        self.ctx[BuildBegemotExecutable._targets_collected] = list(targets)

        # This line can set BuildSystem parameter as a side effect
        wizard_utils.setup_hosts_sdk1(self, trunk_speedup=True)
        YaMakeTask.on_enqueue(self)

    def get_resources(self):
        ret = {}
        for rname in self.ctx[BuildBegemotExecutable._targets_collected]:
            rtype = _resource_by_name(rname)
            base = os.path.basename(rtype.arcadia_build_path)
            ret[rname] = {
                'description': base,
                'resource_path': base,
                'resource_type': rtype,
            }
        return ret

    def get_build_def_flags(self):
        additional_flags = [self.ctx.get(build_params.DefinitionFlags.name, '')]
        targets = set(self.ctx[BuildBegemotExecutable._targets_collected])
        if (
            {br.BEGEMOT_YT_MAPPER.name, br.BEGEMOT_EXECUTABLE.name} & targets
            and self.ctx[build_params.BuildType.name] == 'debug'
        ):
            additional_flags.append('--pic')  # DEVTOOLS-6258, might add but is not guaranteed to
        if br.BEGEMOT_BEGGINS_EXECUTABLE.name in targets:
            additional_flags += ['-DCUDA_VERSION=11.4', '-DCUDNN_VERSION=8.0.5']  # DEVTOOLSSUPPORT-15848
        return '-DSANDBOX_TASK_ID={} {additional_flags}'.format(self.id, additional_flags=' '.join(additional_flags))

    def get_targets(self):
        return [
            _build_path(_resource_by_name(rname).arcadia_build_path)
            for rname in self.ctx[BuildBegemotExecutable._targets_collected]
        ]

    def get_arts(self):
        return [
            {'path': _resource_by_name(rname).arcadia_build_path}
            for rname in self.ctx[BuildBegemotExecutable._targets_collected]
        ]

    def post_build(self, source_dir, output_dir, pack_dir):
        # Проверям передавал ли родительский таск id ресурса, который нужно заполнить
        # TODO: try to use self.parent_resources.get(res_type)
        for rname in self.ctx[BuildBegemotExecutable._targets_collected]:
            logging.debug("Saving resource %s for the parent task", rname)
            rtype = _resource_by_name(rname)
            for resource in channel.sandbox.list_resources(resource_type=rtype, task_id=self.id) or []:
                if self.ctx[consts.SANITIZE]:
                    resource.attributes[consts.SANITIZE] = self.ctx[consts.SANITIZE]

                if self.ctx.get(rname):
                    resource_to_save = channel.sandbox.get_resource(self.ctx[rname])
                    if resource_to_save.is_ready():
                        logging.debug("Resource #%s has already been saved" % self.ctx[rname])
                    else:
                        self.save_parent_task_resource(resource.path, self.ctx[rname])

        wizard_utils.delete_raw_build_output(self.id)

__Task__ = BuildBegemotExecutable
