# -*- coding: utf-8 -*-
import json
import logging
import os
import re

from sandbox.sandboxsdk import task, parameters
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.websearch.begemot.resources as br


def clean_json(string):
    string = re.sub(',[ \t\r\n]+}', '}', string)
    string = re.sub(',[ \t\r\n]+\]', ']', string)
    return string


def use_all_rules(path):
    # Если не указать список правил в запросе, то будут собраны ответы вообще со всех
    with open(path, 'rb') as fd:
        try:
            config = json.loads(clean_json(fd.read()))
        except ValueError:
            return
        for x in config[0]['results']:
            if x['type'] in ('begemot_config', 'begemot_rules'):
                x.pop('requested_rules', None)
    with open(path, 'wb') as fd:
        fd.write(json.dumps(config))


class AllRules(parameters.SandboxBoolParameter):
    name = 'all_rules'
    description = 'Use all rules in Begemot config'
    required = False
    default_value = False


class BuildBegemotLightTestConfig(task.SandboxTask):

    type = 'BUILD_BEGEMOT_LIGHT_TEST_CONFIG'
    execution_space = 64

    input_parameters = [
        build_params.ArcadiaUrl,
        AllRules
    ]

    def on_enqueue(self):
        self.create_resource(
            resource_type=br.BEGEMOT_CONFIG_WITH_CGI,
            description=self.description,
            resource_path='config',
        )

    def on_execute(self):
        parsed_url = Arcadia.parse_url(self.ctx[build_params.ArcadiaUrl.name])

        modified_config_path = os.path.join(parsed_url.path, br.BEGEMOT_CONFIG_WITH_CGI.arcadia_build_path)
        modified_config_url = Arcadia.replace(self.ctx[build_params.ArcadiaUrl.name], path=modified_config_path)
        logging.debug('Modified url is: {}'.format(modified_config_url))

        resources = channel.sandbox.list_resources(resource_type=br.BEGEMOT_CONFIG_WITH_CGI, task_id=self.id)

        for resource in resources:
            logging.debug('Current resource path is: {}'.format(resource.path))
            Arcadia.export(modified_config_url, resource.path)

            if self.ctx[AllRules.name]:
                use_all_rules(os.path.join(resource.path, 'test', 'cgi_contexts', 'test_context.json'))

            # Проверяем передавал ли родительский таск id ресурса, который нужно заполнить
            if self.ctx.get(br.BEGEMOT_CONFIG_WITH_CGI.name):
                self.save_parent_task_resource(resource.path, self.ctx[br.BEGEMOT_CONFIG_WITH_CGI.name])


__Task__ = BuildBegemotLightTestConfig
