# -*- coding: utf-8 -*-
import os
import logging
import datetime

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import parameters, svn
from sandbox.projects import resource_types
from sandbox.projects.common.build.YaMake import YaMakeTask
import sandbox.projects.common.build.parameters as build_params


POST_BUILD_PARAMS_GROUP = 'Post build params'


class StripBinaryParameter(parameters.SandboxBoolParameter):
    name = 'strip_binary'
    description = 'Strip binary'
    group = POST_BUILD_PARAMS_GROUP
    default_value = True


class ResourceTtlParameter(parameters.SandboxIntegerParameter):
    name = 'resource_ttl'
    description = 'Resource ttl (-1 for inf)'
    group = POST_BUILD_PARAMS_GROUP
    default_value = 30


class ResourceTagParameter(parameters.SandboxStringParameter):
    name = 'resource_tag'
    description = 'Resource tag (for filtration)'
    group = POST_BUILD_PARAMS_GROUP
    default_value = ''


class ArcadiaRevisionOverwriteParameter(parameters.SandboxStringParameter):
    name = 'arcadia_revision_overwrite'
    description = 'Arcadia revision (overwrite value getting from svn url)'
    group = build_params.ArcadiaUrl.group


class BuildBlenderLibblndr(YaMakeTask):
    """
        Build blender python bindings library (libblndr.so).
    """

    type = 'BUILD_BLENDER_LIBBLNDR'

    input_parameters = [
        build_params.ArcadiaUrl,
        ArcadiaRevisionOverwriteParameter,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaryParameter,
        ResourceTtlParameter,
        ResourceTagParameter
    ] + build_params.get_aapi_parameters()

    client_tags = YaMakeTask.client_tags & ctc.Tag.Group.LINUX

    @property
    def resource_name(self):
        return os.path.basename(self.resource_path)

    @property
    def resource_path(self):
        return 'yweb/blender/lib/libblndr/libblndr.so'

    def get_targets(self):
        return [os.path.dirname(self.resource_path)]

    def get_build_def_flags(self):
        return '-DWERROR=no'

    def get_arts(self):
        return [{'path': self.resource_path}]

    def get_resources(self):
        return {
            self.resource_name: {
                'description': self.descr,
                'resource_type': resource_types.BLENDER_LIBBLNDR,
                'resource_path': self.resource_name
            }
        }

    # instead of set attr and ttl we can release resource (create_release) and clear ttl for old ones
    def get_resources_attrs(self):
        ttl = self.ctx.get(ResourceTtlParameter.name)
        attrs = {
            'tag': self.ctx.get(ResourceTagParameter.name),
            'ttl': ttl if ttl > 0 else 'inf',
            'build_date': datetime.date.today().strftime('%Y%m%d'),
        }
        if self.ctx.get(StripBinaryParameter.name):
            attrs['stripped'] = 1
        return {self.resource_name: attrs}

    def post_build(self, source_dir, output_dir, pack_dir):
        if self.ctx.get(StripBinaryParameter.name):
            cmd = 'strip ' + os.path.join(pack_dir, self.resource_name)
            self._subprocess(cmd, check=True, wait=True, log_prefix='strip_binary')

    # hack for work with overwrite revision parameter (need in nirvana jobs)
    def pre_execute(self):
        rev = self.ctx.get(ArcadiaRevisionOverwriteParameter.name)
        if rev:
            rev = int(rev)
            arcadia_url = self.ctx.get(build_params.ArcadiaUrl.name)
            parsed_url = svn.Arcadia.parse_url(arcadia_url)
            logging.info('overwrite svn url %s revision by direct parameter value %s', parsed_url.revision, rev)
            if parsed_url.revision is not None:
                arcadia_url = arcadia_url.rsplit('@', 1)[0]
            self.ctx[build_params.ArcadiaUrl.name] = arcadia_url + '@' + str(rev)
        YaMakeTask.pre_execute(self)


__Task__ = BuildBlenderLibblndr
