# -*- coding: utf-8 -*-
"""
* Build google-breakpad from source tarball
* Create libbreakpad_init.so to initialize breakpad via LD_PRELOAD

Author: alonger@
"""

import os
import time
import shutil

from sandbox.projects import resource_types
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector


class BreakpadTarball(ResourceSelector):
    name = 'breakpad_tgz_resource_id'
    # obtained from https://code.google.com/p/google-breakpad/
    description = 'breakpad source code tarball'
    resource_type = 'OTHER_RESOURCE'
    required = True


class BuildBreakpadInit(SandboxTask):
    """
    Builds Breakpad Init shared library
    """
    type = 'BUILD_BREAKPAD'

    input_parameters = (
        BreakpadTarball,
    )

    def arcadia_info(self):
        """
            Получение информации о задаче при релизе

            :return список из трёх значений revision, tag, branch
        """
        return None, time.strftime("%Y%m%d_%H%M%S"), None

    def on_execute(self):
        Arcadia.checkout(
            Arcadia.trunk_url('devtools/breakpad_init'),
            self.abs_path('breakpad_init')
        )

        os.chdir(self.abs_path('breakpad_init'))

        src_gz = 'breakpad.tar.gz'

        shutil.copy(
            self.sync_resource(self.ctx['breakpad_tgz_resource_id']),
            self.abs_path('breakpad_init', src_gz)
        )

        run_process('tar -xvzf ' + src_gz, log_prefix='extract-breakpad')
        run_process('rm ' + src_gz)

        run_process('chmod +x build_breakpad.sh')
        run_process('./build_breakpad.sh', log_prefix='make')

        self.create_resource(
            'Breakpad Init library',
            self.abs_path('breakpad_init', 'libbreakpad_init.so'),
            resource_types.BREAKPAD_INIT_SO)

        self.create_resource(
            'Breakpad Minidump Stackwalk',
            self.abs_path('breakpad_init', 'google-breakpad/src/processor/minidump_stackwalk'),
            resource_types.BREAKPAD_MINIDUMP_STACKWALK)

        self.create_resource(
            'Breakpad Symbols Dumper',
            self.abs_path('breakpad_init', 'google-breakpad/src/tools/linux/dump_syms/dump_syms'),
            resource_types.BREAKPAD_SYMBOLS_DUMPER)


__Task__ = BuildBreakpadInit
