# coding: utf-8

import os
import logging

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import add_executable_permissions_for_path


class BuildBsconfigGitTag(SandboxStringParameter):
    """
        Путь в svn до папки с bsconfig
    """
    name = 'build_bsconfig_git_tag'
    description = 'Bsconfig git tag name:'
    default_value = '6.3-0stable'


class BuildBsconfig(SandboxTask):
    """
        Выкачиваем bsconfig, запускаем тесты, сохраняем его как ресурс
    """

    type = 'BUILD_BSCONFIG'

    input_parameters = (BuildBsconfigGitTag, )

    # пока что указываем явно пользователя sandbox-а
    BSCONFIG_GIT_REPO = 'https://git.qe-infra.yandex-team.ru/scm/search_infra/bsconfig.git'

    def on_execute(self):
        git_bsconfig_folder = self.path('git_bsconfig_src')
        logging.info('Download bsconfig from {repo}, tag {tag} to {folder}'.format(
            repo=self.BSCONFIG_GIT_REPO, tag=self.ctx['build_bsconfig_git_tag'], folder=git_bsconfig_folder,))

        run_process(['git', 'clone', self.BSCONFIG_GIT_REPO, git_bsconfig_folder], log_prefix='git_clone_bsconfig')

        os.chdir(git_bsconfig_folder)

        run_process(['git', 'checkout', 'tags/{tag}'.format(tag=self.ctx['build_bsconfig_git_tag'])],
                    log_prefix='git_checkout_bsconfig_tag')

        bsconfig_path = self.path('bsconfig')
        # перемещаем выкачанный файл bsconfig-а в корень диретории задачи
        os.rename(os.path.join(git_bsconfig_folder, 'bsconfig'), bsconfig_path)
        add_executable_permissions_for_path(bsconfig_path)

        logging.info('Run bsconfig tests for file {bsconfig_file}.'.format(bsconfig_file=bsconfig_path))
        run_process(['perl', '-c', bsconfig_path], log_prefix='test_bsconfig')

        self.create_resource(
            'bsconfig tag {tag}'.format(tag=self.ctx['build_bsconfig_git_tag']),
            bsconfig_path,
            resource_types.BSCONFIG,
            arch='any'
        )

    def arcadia_info(self):
        return None, self.ctx['build_bsconfig_git_tag'], None


__Task__ = BuildBsconfig
