# coding: utf-8

import os
import shutil
import logging
import tarfile
import time

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.utils import get_bsconfig

from sandbox.projects.common.nanny import nanny


class ResourceDescription(SandboxStringParameter):
    name = 'description'
    description = 'Clusterstate build description'
    default_value = 'Clusterstate trunk'


class ArcadiaURL(SandboxSvnUrlParameter):
    name = 'arcadia_url'
    description = 'SVN URL (arcadia)'
    default_value = Arcadia.trunk_url() + '/infra/clusterstate'


class CreateShard(SandboxBoolParameter):
    name = 'create_shard'
    description = 'Create bsconfig shard (call shard_init):'
    default_value = False


class BuildClusterstate(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Build clusterstate.
    """
    type = 'BUILD_CLUSTERSTATE'
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = [
        ResourceDescription,
        ArcadiaURL,
        CreateShard,
    ]

    CHECKOUT_PATH = 'cluster_state'
    TGZ_PATH = 'clusterstate.tar.gz'

    def _create_shard(self):
        shard_name = 'clusterstate-{0}'.format(time.strftime("%Y%m%d_%H%M%S"))
        self.ctx['shard_name'] = shard_name
        shard_path = self.path(shard_name)
        remove_path(shard_path)
        os.mkdir(shard_name)
        shutil.copy(self.TGZ_PATH, shard_name)
        with open(os.path.join(shard_name, 'shard.conf'), 'w') as f:
            f.write('%install\n')
            f.write('tar -xzf {}\n'.format(self.TGZ_PATH))
            f.write('/bin/bash get_ve.sh\n')
        self.create_resource(
            description='clusterstate shard',
            resource_path=shard_path,
            resource_type=resource_types.CLUSTERSTATE_SHARD,
        )
        run_process(
            ['perl', get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config",
            work_dir=self.path()
        )

    def _checkout(self):
        self.local_dir = os.path.realpath(self.CHECKOUT_PATH)
        self.url = self.ctx.get(ArcadiaURL.name)
        Arcadia.export(self.url, self.local_dir)

    def _make_resource(self, virtualenv_path):
        logging.info('creating tgz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
        self.create_resource(
            description=self.ctx['description'],
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.CLUSTERSTATE
        )

    def on_execute(self):
        self._checkout()

        env = dict(os.environ)
        env['PYTHON'] = '/skynet/python/bin/python'
#        run_process(
#            ['bash', os.path.join(cs_path, 'build.sh'), 'dev'],
#            work_dir=cs_path,
#            log_prefix='build_cs',
#            shell=True,
#            environment=env)
        self._make_resource(self.local_dir)
        if self.ctx.get('create_shard'):
            self._create_shard()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = BuildClusterstate
